import re
from functools import partial
from collections import namedtuple

from sandbox import sdk2
from sandbox.common.errors import TaskError

from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
        import validate_has_exact_svn_revision
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
        import is_branch_up_to_date, BranchDependentValue, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK


_BUNDLES = [
    'auth',
    'recording',
    'datasync',
    'mapkit',
    'directions',
    'places',
    'search',
    'transport',
    'bookmarks',
    'push',
    'mrc',
]

_PLATFORM_BUNDLES = [
    'styling_car_navigation'
]

_PLATFORM_TO_MAKE_SUBTASK_TYPE = {
    'android': 'MAPS_MOBILE_MAKE_ARTIFACTS',
    'darwin': 'MAPS_MOBILE_MAKE_ARTIFACTS',
    'ios': 'MAPS_MOBILE_BUILD_FRAMEWORK',
    'linux': 'MAPS_MOBILE_MAKE_ARTIFACTS',
}
_PLATFORM_TO_MAKE_PLATFORM_SUBTASK_TYPE = {
    'android': 'MAPS_MOBILE_BUILD_ANDROID_AAR',
    'ios': 'MAPS_MOBILE_MAKE_PLATFORM_IOS_FRAMEWORK',
}
_MAKE_DEV_BUNDLE_SUBTASK_TYPE = 'MAPS_MOBILE_MAKE_DEV_BUNDLE'
_GENERATE_DOCUMENTATION_SUBTASK_TYPE = {
    'ios': 'MAPS_MOBILE_GENERATE_JAZZY_DOCS'
}
_PLATFORM_TO_RELEASE_SUBTASK_TYPE = {
    'android': 'MAPS_MOBILE_RELEASE_MAVEN',
    'ios': 'MAPS_MOBILE_RELEASE_COCOA_POD',
}

PlatformAndroidConfig = namedtuple(
    'PlatformAndroidConfig',
    [
        'group_id',
        'artifact_id',
        'src_path',
        'jdk_resource',
        'android_sdk_resource',
        'gradle_resource',
        'maven_repo_resource',
        'keystore',
   ])

_FIRST_BRANCH_WITH_JDK_30 = '2021091700'  # fake branch
_ANDROID_SDK_RESOURCE = BranchDependentValue({_FIRST_BRANCH_WITH_JDK_30: 2431170592}, 1385186850)
_FIRST_BRANCH_WITH_GRADLE_6 = '2021111316' # fake branch
_GRADLE_RESOURCE = BranchDependentValue({_FIRST_BRANCH_WITH_GRADLE_6: 2566840956}, 1368085369)


_PLATFORM_ANDROID_PROJECTS = {
    'com.yandex.mapkit.styling.carnavigation': PlatformAndroidConfig(
        group_id='com.yandex.mapkit.styling',
        artifact_id='carnavigation',
        src_path='maps/mobile/libs/navigation_resource_provider/android',
        jdk_resource=810373574,
        android_sdk_resource=_ANDROID_SDK_RESOURCE,
        gradle_resource=_GRADLE_RESOURCE,
        maven_repo_resource=2586526945,
        keystore="sec-01dvx8g7d7bbxzff6d61t44jek",
        ),
    'com.yandex.mapkit.styling.roadevents': PlatformAndroidConfig(
        group_id='com.yandex.mapkit.styling',
        artifact_id='roadevents',
        src_path='maps/mobile/libs/road_events_layer_style_provider/android',
        jdk_resource=810373574,
        android_sdk_resource=_ANDROID_SDK_RESOURCE,
        gradle_resource=_GRADLE_RESOURCE,
        maven_repo_resource=2586526945,
        keystore="sec-01dvx8g7d7bbxzff6d61t44jek",
        ),
    'com.yandex.mapkit.styling.transportnavigation': PlatformAndroidConfig(
        group_id='com.yandex.mapkit.styling',
        artifact_id='transportnavigation',
        src_path='maps/mobile/libs/transport/navigation_resource_provider/android',
        jdk_resource=810373574,
        android_sdk_resource=_ANDROID_SDK_RESOURCE,
        gradle_resource=_GRADLE_RESOURCE,
        maven_repo_resource=2586526945,
        keystore="sec-01dvx8g7d7bbxzff6d61t44jek",
        ),
}

_LIBRARY_TO_PLATFORM_ANDROID_PROJECTS = {
    'styling_car_navigation': {'com.yandex.mapkit.styling.carnavigation'},
    'styling_road_events': {'com.yandex.mapkit.styling.roadevents'},
    'styling_transport_navigation': {'com.yandex.mapkit.styling.transportnavigation'},
}

_IOS_FRAMEWORK_NAME = 'YandexMapsMobile'

_IOS_LIBRARY_TO_SYSTEM_FRAMEWORKS = {
    'runtime': {'Foundation', 'UIKit', 'OpenGLES', 'CoreFoundation', 'Security', 'CoreTelephony',
                'CoreLocation', 'QuartzCore', 'CoreGraphics', 'SystemConfiguration'},
    'auth': {'SystemConfiguration', 'MobileCoreServices', 'QuartzCore', 'CoreText'},
}

PlatformIosConfig = namedtuple(
    'PlatformIosConfig',
    [
        'framework_name',
        'bundle_name',
        'src_path',
        'framework_dependencies',
        'library_dependencies',
    ])

_PLATFORM_IOS_PROJECTS = {
    'YMKStylingCarNavigation': PlatformIosConfig(
        framework_name='YMKStylingCarNavigation',
        bundle_name='YMKCarNavigationResources',
        src_path='maps/mobile/libs/navigation_resource_provider/ios',
        framework_dependencies=['Foundation', 'UIKit', 'YandexMapsMobile'],
        library_dependencies=['c++'],
        ),
    'YMKStylingRoadEvents': PlatformIosConfig(
        framework_name='YMKStylingRoadEvents',
        bundle_name='YMKStylingRoadEventsBundle',
        src_path='maps/mobile/libs/road_events_layer_style_provider/ios',
        framework_dependencies=['Foundation', 'UIKit', 'YandexMapsMobile'],
        library_dependencies=['c++'],
        ),
    'YMKStylingTransportNavigation': PlatformIosConfig(
        framework_name='YMKStylingTransportNavigation',
        bundle_name='YMKTransportNavigationResources',
        src_path='maps/mobile/libs/transport/navigation_resource_provider/ios/YMKStylingTransportNavigation',
        framework_dependencies=['Foundation', 'UIKit', 'YandexMapsMobile'],
        library_dependencies=['c++'],
        ),

}

_LIBRARY_TO_PLATFORM_IOS_PROJECTS = {
    'styling_car_navigation': {'YMKStylingCarNavigation'},
    'styling_road_events': {'YMKStylingRoadEvents'},
    'styling_transport_navigation': {'YMKStylingTransportNavigation'},
}


class MapsMobileExportArtifactsBase(sdk2.Task):
    """
    Base task for common code of MAPS_MOBILE_EXPORT_ARTIFACTS
    and MAPS_MOBILE_EXPORT_NAVI_ARTIFACTS
    """

    _FIRST_BRANCH_WITH_PLATFORM_PROJECTS = '2021080614'

    class Parameters(sdk2.Task.Parameters):

        is_snapshot = False

        arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Arcadia URL with a required revision:',
                description='An example: arcadia:/arc/trunk/arcadia@6150000',
                required=True)

        custom_release_version = sdk2.parameters.String(
                'Custom release version',
                description='Leave empty for autogeneration.'
                            ' Do not add "-snapshot" to the version: the Task'
                            ' will append it if necessary.',
                )

        generate_doc = sdk2.parameters.Bool(
                'Generate documentation',
                description='Build documentation for Java/ObjC code',
                default=False)

        is_snapshot = sdk2.parameters.Bool('Is snapshot', default=False)

        is_ios_dynamic = sdk2.parameters.Bool('Is iOS dynamic', default=False)
        with is_ios_dynamic.value[False]:
            mangle_symbols = sdk2.parameters.Bool('Mangle symbols', default=True)

        with sdk2.parameters.CheckGroup('Libraries') as libraries:
            for bundle in _BUNDLES:
                libraries.values[bundle] = bundle

        with sdk2.parameters.CheckGroup('Platform Libraries') as platform_libraries:
            for bundle in _PLATFORM_BUNDLES:
                platform_libraries.values[bundle] = bundle


        with sdk2.parameters.Output:
            release_version = sdk2.parameters.String('Release version')
            android_artifacts = sdk2.parameters.Resource('Artifacts')
            framework_zip = sdk2.parameters.Resource('Framework zip')
            documentation = sdk2.parameters.Resource('Raw documentation')

    def on_execute(self):
        self._set_up()
        self._make_artifacts()
        if is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_PLATFORM_PROJECTS):
            if self.Parameters.platform == 'ios':
                self._make_platform_ios_projects(sdk2.Resource[self.Context.framework_tar_gz])
            if self.Parameters.platform == 'android':
                self._make_platform_android_projects(self.Parameters.android_artifacts)


        if (self.Parameters.generate_doc
                and self.Parameters.platform in _GENERATE_DOCUMENTATION_SUBTASK_TYPE):
            self._generate_documentation()

        if self.Parameters.platform in _PLATFORM_TO_RELEASE_SUBTASK_TYPE:
            self._release()

    def _set_up(self):
        validate_has_exact_svn_revision(self.Parameters.arcadia_url)
        self._set_release_version()

    def _set_release_version(self):
        release_version = self.Parameters.custom_release_version or self._release_version_from_arcadia_url()
        if not self.Parameters.is_navi and self.Parameters.is_snapshot:
            release_version += '-snapshot'
        self.Parameters.release_version = release_version

    def _release_version_from_arcadia_url(self):
        regex = re.compile('^arcadia:/arc/(branches/maps-mobile-releases/)?([^/]*)/arcadia@([0-9]*)$')
        match = regex.match(self.Parameters.arcadia_url)
        if not match:
            raise TaskError('Can generate Release Version only for ArcadiaUrl matching regex: '
                              '"{pattern}".'.format(pattern=regex.pattern))
        date = match.group(2) if match.group(2) != "trunk" else 0
        revision = match.group(3)
        return ('{date}.{revision}-{task_id}'
                .format(date=date, revision=revision, task_id=self.id))

    def _make_artifacts(self):
        subtask_type = (_PLATFORM_TO_MAKE_SUBTASK_TYPE[self.Parameters.platform]
                        if self.Parameters.platform
                        else _MAKE_DEV_BUNDLE_SUBTASK_TYPE)
        parameters = {
            'arcadia_url': self.Parameters.arcadia_url,
            'release_version': self.Parameters.release_version,
            'libraries': self.Parameters.libraries,
        }
        if self.Parameters.platform in ['android', 'darwin', 'linux']:
            parameters['platform'] = self.Parameters.platform
        if self.Parameters.platform == 'android':
            parameters['generate_doc'] = self.Parameters.generate_doc
        if self.Parameters.platform == 'ios':
            parameters.update({
                'ya_make_arcadia_dirname': (
                    'maps/mobile/bundle/dynamic'
                    if self.Parameters.is_ios_dynamic
                    else 'maps/mobile/bundle/recursive_library'
                ),
                'is_dynamic': self.Parameters.is_ios_dynamic,
                'mangle_symbols': self.Parameters.mangle_symbols,
            })
        if self.Parameters.is_navi:
            parameters['is_upload'] = self.Parameters.is_upload
        if self.Parameters.is_navi and self.Parameters.platform:
            parameters['is_navi'] = True
        SubtaskRunner(self).run_and_handle_result(subtask_type, parameters, self._handle_make_artifacts_result)

    def _handle_make_artifacts_result(self, subtask):
        if self.Parameters.platform == 'android':
            self.Parameters.android_artifacts = subtask.Parameters.artifacts
            if self.Parameters.generate_doc:
                self.Parameters.documentation = subtask.Parameters.documentation
        if self.Parameters.platform == 'ios':
            self.Parameters.framework_zip = subtask.Parameters.framework_zip
            if is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_PLATFORM_PROJECTS):
                self.Context.framework_tar_gz = subtask.Parameters.framework_tar_gz.id

    def _make_platform_android_project(self, project, bundle_resource):
        subtask_type = _PLATFORM_TO_MAKE_PLATFORM_SUBTASK_TYPE['android']
        project_config = _PLATFORM_ANDROID_PROJECTS[project]
        arcadia_url_tokens = self.Parameters.arcadia_url.split('@')
        subtask_parameters = {
            'arcadia_url': arcadia_url_tokens[0],
            'arcadia_revision': arcadia_url_tokens[1],
            'prebuilt_bundle': True,
            'bundle_resource': bundle_resource,
            'jdk_resource': project_config.jdk_resource,
            'android_sdk_resource': project_config.android_sdk_resource.get(arcadia_url_tokens[0]),
            'gradle_resource': project_config.gradle_resource.get(arcadia_url_tokens[0]),
            'maven_repo_resource': project_config.maven_repo_resource,
            'keystore': project_config.keystore,
            'project_src_path': project_config.src_path,
            'build_variant': 'release',
            'group_id': project_config.group_id,
            'artifact_id': project_config.artifact_id,
            'artifact_version': self.Parameters.release_version,
            'is_navi': self.Parameters.is_navi,
        }
        SubtaskRunner(self).run_and_handle_result(
            subtask_type,
            subtask_parameters,
            result_callback=partial(self._handle_make_platform_android_project_result, project=project),
            tag=project,
            require_successful=True
            )

    def _handle_make_platform_android_project_result(self, subtask, project):
        resource = sdk2.Resource['MAPS_MOBILE_AAR'].find(task=subtask).first()
        if not resource:
            raise TaskError('Could not find YA_PACKAGE resource')
        self.Context.platform_android_results[project] = resource.id

    def _make_platform_android_projects(self, bundle_resource):
        if not self.Context.platform_android_results:
            self.Context.platform_android_results = {}
        projects = {project
            for library in self.Parameters.platform_libraries
            for project in _LIBRARY_TO_PLATFORM_ANDROID_PROJECTS.get(library, [])}
        for project in projects:
            self._make_platform_android_project(project, bundle_resource)

    def _make_platform_ios_project(self, project, bundle_resource):
        subtask_type = _PLATFORM_TO_MAKE_PLATFORM_SUBTASK_TYPE['ios']
        project_config = _PLATFORM_IOS_PROJECTS[project]
        subtask_parameters = {
            'arcadia_url': self.Parameters.arcadia_url,
            'release_version': self.Parameters.release_version,
            'maps_mobile_bundle_resource': bundle_resource,
            'framework_project': project_config.framework_name,
            'bundle_project': project_config.bundle_name,
            'is_dynamic': self.Parameters.is_ios_dynamic,
            'project_src_path': project_config.src_path,
            'is_navi': self.Parameters.is_navi
        }
        SubtaskRunner(self).run_and_handle_result(
            subtask_type,
            subtask_parameters,
            result_callback=partial(self._handle_make_platform_ios_project_result, project=project),
            tag=project,
            require_successful=True
            )

    def _handle_make_platform_ios_project_result(self, subtask, project):
        self.Context.platform_ios_results[project] = subtask.Parameters.framework_zip.id

    def _make_platform_ios_projects(self, bundle_resource):
        if not self.Context.platform_ios_results:
            self.Context.platform_ios_results = {}
        projects = {project
            for library in self.Parameters.platform_libraries
            for project in _LIBRARY_TO_PLATFORM_IOS_PROJECTS.get(library, [])}
        for project in projects:
            self._make_platform_ios_project(project, bundle_resource)

    def _generate_documentation(self):
        subtask_type = _GENERATE_DOCUMENTATION_SUBTASK_TYPE[self.Parameters.platform]
        parameters = {
                'framework_zip': self.Parameters.framework_zip,
                'module_version': self.Parameters.release_version,
                'arcadia_url': self.Parameters.arcadia_url,
        }
        SubtaskRunner(self).run_and_handle_result(
                subtask_type, parameters, self._handle_generate_documentation_result,
                require_successful=False)

    def _handle_generate_documentation_result(self, subtask):
        if self.Parameters.platform == 'ios':
            self.Parameters.documentation = subtask.Parameters.documentation

    def _release_android_platform_results(self):
        subtask_type = _PLATFORM_TO_RELEASE_SUBTASK_TYPE['android']
        for project, resource_id in self.Context.platform_android_results.iteritems():
            project_config = _PLATFORM_ANDROID_PROJECTS[project]
            subtask_parameters = {
                'aar_resource': sdk2.Resource[resource_id],
                'aar_base_name': project,
                'version': self.Parameters.release_version,
                'group_id': project_config.group_id,
                'artifact_id': project_config.artifact_id,
                'is_snapshot': self.Parameters.is_snapshot,
            }
            SubtaskRunner(self).run_and_handle_result(subtask_type, subtask_parameters, tag=project + '_release')

    def _release_ios_platform_results(self):
        subtask_type = _PLATFORM_TO_RELEASE_SUBTASK_TYPE['ios']
        for project, resource_id in self.Context.platform_ios_results.iteritems():
            project_config = _PLATFORM_IOS_PROJECTS[project]
            subtask_parameters =  {
               'framework_zip': sdk2.Resource[resource_id],
               'framework_name': project,
               'release_version': self.Parameters.release_version,
               'is_snapshot_repo': self.Parameters.is_snapshot,
               'framework_dependencies': project_config.framework_dependencies,
               'library_dependencies': project_config.library_dependencies,
               'framework_dir': self._framework_dir(project_config.framework_name),
               'bundle_dir': self._bundle_dir(project_config.framework_name, project_config.bundle_name),
            }
            SubtaskRunner(self).run_and_handle_result(subtask_type, subtask_parameters, tag=project + '_release')

    def _release(self):
        subtask_type = _PLATFORM_TO_RELEASE_SUBTASK_TYPE[self.Parameters.platform]
        parameters = (
            self._cocoa_main_bundle_parameters(_IOS_FRAMEWORK_NAME)
            if self.Parameters.platform == 'ios'
            else self._maven_main_bundle_parameters()
        )
        SubtaskRunner(self).run_and_handle_result(subtask_type, parameters, tag='main')

        if self.Parameters.platform == 'android' and is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_PLATFORM_PROJECTS):
            self._release_android_platform_results()

        if self.Parameters.platform == 'ios' and is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_PLATFORM_PROJECTS):
            self._release_ios_platform_results()

    def _framework_dir(self, framework_name):
        if (self.Parameters.is_ios_dynamic or
                is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK)):
            return framework_name + '.xcframework'
        return framework_name + '.framework'

    def _bundle_dir(self, framework_name, bundle_name, is_inner=False):
        if (self.Parameters.is_ios_dynamic or not is_inner or 
                is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK)):
            return bundle_name + '.bundle'
        return '{framework_name}.framework/Versions/A/Resources/{bundle_name}.bundle'.format(
            framework_name=framework_name,
            bundle_name=bundle_name)

    def _cocoa_main_bundle_parameters(self, framework_name):
        return {
                   'framework_zip': self.Parameters.framework_zip,
                   'framework_name': framework_name,
                   'release_version': self.Parameters.release_version,
                   'is_snapshot_repo': self.Parameters.is_snapshot,
                   'framework_dependencies': list({framework
                       for lib in ['runtime'] + self.Parameters.libraries
                       for framework in _IOS_LIBRARY_TO_SYSTEM_FRAMEWORKS.get(lib, [])}),
                   'library_dependencies': ['resolv', 'c++'],
                   'framework_dir': self._framework_dir(framework_name),
                   'bundle_dir': self._bundle_dir(framework_name, framework_name, True),
               }

    def _maven_main_bundle_parameters(self):
        return {
                   'group_id': 'com.yandex.maps',
                   'artifact_id': 'mobile',
                   'version': self.Parameters.release_version,
                   'aar_resource': self.Parameters.android_artifacts,
                   'aar_base_name': 'maps.mobile',
                   'is_snapshot': self.Parameters.is_snapshot,
               }
