import glob
import logging
import os
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import shutil
import yaml

from sandbox import sdk2
from sandbox.common.errors import TaskError

import sandbox.common.types.client as ctc
from sandbox.projects.maps.mobile.MapsMobileResources \
        import MapsMobileJazzyDocumentation, \
        MapsMobileFrameworkZip


# We won't generate documentation very often, so we make TTL infinite
_JAZZY_DOCS_TTL = 'inf'


class MapsMobileGenerateJazzyDocs(sdk2.Task):
    """ Task for generating Swift/Objc documentation with Jazzy. """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_MAPS_MOBILE_DARWIN & ctc.Tag.USER_MAPS_MOBILE_DARWIN

    class Parameters(sdk2.Task.Parameters):

        framework_zip = sdk2.parameters.Resource(
                'Framework zip',
                required=True,
                resource_type=MapsMobileFrameworkZip)

        arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Arcadia URL with revision against which the framework was built:',
                description='An example: arcadia:/arc/trunk/arcadia@6150000',
                required=True)

        module_version = sdk2.parameters.String(
                'Framework version', required=True)

        with sdk2.parameters.Output:
            documentation = sdk2.parameters.Resource('Raw Jazzy documentation')

    def on_execute(self):
        self._framework_zip_path = str(
                sdk2.ResourceData(self.Parameters.framework_zip).path)
        self._working_directory = os.path.abspath('YandexMapsMobile.framework')

        headers_dir = os.path.join(self._working_directory, 'Headers')

        self._umbrella_header_path = os.path.join(
                self._working_directory,
                'Headers/YMUmbrella.h')
        self._output_path = os.path.join(
                self._working_directory,
                'YandexMapsMobile-{}-jazzy-doc'
                    .format(self.Parameters.module_version))

        self._unpack_framework()
        self._headers = glob.glob(os.path.join(headers_dir, '*.h'))
        self._generate_umbrella_header()
        self._move_headers()
        self._generate_docs()
        self._make_resource_ready()

    def _unpack_framework(self):
        sdk2.helpers.subprocess.check_call(
            [
                'unzip',
                '-q', self._framework_zip_path])

    def _generate_umbrella_header(self):
        """
        Take filenames from YandexMapsMobile.framework/Headers
        and put them under #import in this header
        """
        with open(self._umbrella_header_path, 'w') as umbrella_header:
            for header in self._headers:
                umbrella_header.write(
                    '#import <YandexMapsMobile/{}>\n'.format(os.path.basename(header)))

    def _move_headers(self):
        """
        Move non-internal headers from Headers/ to Headers/YandexMapsMobile
        to bypass invalid Jazzy compiling attempts
        """
        temp_headers_dir = os.path.join(
                self._working_directory,
                'Headers/YandexMapsMobile')
        if os.path.isdir(temp_headers_dir):
            raise TaskError('Failed to create a temporary headers directory: {} '
                              'already exists!'.format(temp_headers_dir))
        os.mkdir(temp_headers_dir)
        for header in self._headers:
            shutil.move(header, temp_headers_dir)

    def _get_xcode_version(self):
        path_to_build_config = os.path.join('maps', 'mobile', '.build_config.yaml')
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            with open(os.path.join(arcadia, path_to_build_config)) as f:
                return yaml.safe_load(f)["ios"]["xcode_version"]

    def _get_developer_env(self):
        xcode_version = self._get_xcode_version()
        developer_env = os.environ
        developer_env['DEVELOPER_DIR'] = os.path.join(
            '/Applications',
            'Xcode{}.app'.format(xcode_version),
            'Contents',
            'Developer'
            )

    def _generate_docs(self):
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger("jazzy_doc_generation")) as pl:
            sdk2.helpers.subprocess.check_call(
                [
                    'jazzy',
                    '--objc',
                    '--clean',
                    '--module-version', self.Parameters.module_version,
                    '--author', 'YandexMapsMobile',
                    '--umbrella-header', self._umbrella_header_path,
                    '--framework-root', self._working_directory,
                    '--sdk', 'iphonesimulator',
                    '--source-directory', self._working_directory,
                    '--output', self._output_path,
                    '--separate-global-declarations'],
                cwd=self._working_directory,
                env=self._get_developer_env(),
                stdout=pl.stdout,
                stderr=pl.stderr)

    def _make_resource_ready(self):
        self.Parameters.documentation = MapsMobileJazzyDocumentation(
            self,
            'Raw Jazzy docummentation',
            self._output_path,
            ttl=_JAZZY_DOCS_TTL,
            backup_task=True)
        sdk2.ResourceData(self.Parameters.documentation).ready()
