from collections import namedtuple
import os
import re
import tarfile

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import build

from sandbox.projects.maps.mobile.MapsMobileMakePlatformArtifactsBase \
        import MapsMobileMakePlatformArtifactsBase
from sandbox.projects.maps.mobile.MapsMobileResources \
        import MapsMobileExportArtifactsPkgJson, \
        MapsMobileJavadocDocumentation
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date
from sandbox.projects.maps.mobile.utils.yt_store_parameters import yt_store_parameters
from sandbox.projects.maps.mobile.utils.resource_helpers import (
    apply_navi_export_parameters,
    get_ya_package_resource,
    )


OsConfig = namedtuple(
        'OsConfig',
        [
            'dynamic_library_extensions',
            'ya_package_tarball_task_type',
            'qt_resource',
        ])
_OS_TO_CONFIG = {
    'android': OsConfig(
        dynamic_library_extensions=['so'],
        ya_package_tarball_task_type='MAPS_MOBILE_YA_PACKAGE_LINUX_XENIAL',
        qt_resource=None,
        ),
    'darwin': OsConfig(
        dynamic_library_extensions=['dylib', 'dSYM'],
        ya_package_tarball_task_type='MAPS_MOBILE_YA_PACKAGE_DARWIN',
        qt_resource=1167023706,
        ),
    'linux': OsConfig(
        dynamic_library_extensions=['so'],
        ya_package_tarball_task_type='MAPS_MOBILE_YA_PACKAGE_LINUX_XENIAL',
        qt_resource=1165641810,
        ),
}
_OLD_BUNDLE_AAR_REGEX = re.compile('20200([1-5]|603|611)')
_FIRST_BRANCH_WITH_LOCAL_SO_FROM_AAR = '2020100119'
_FIRST_BRANCH_WITH_AAR_NO_STRIP = '2021042919'
_FIRST_BRANCH_WITH_REDUCED_DEBUGINFO = '2021092300'
_FIRST_BRANCH_WITH_NDK23 = '2021111118'
_JAVADOC_DOCS_TTL = 'inf'  # We won't generate documentation very often, so we make TTL infinite.


class MapsMobileMakeArtifacts(MapsMobileMakePlatformArtifactsBase):
    ''' Task for packaging artifacts. '''

    class Parameters(MapsMobileMakePlatformArtifactsBase.Parameters):
        with sdk2.parameters.String('Platform', required=True) as platform:
            for os_ in _OS_TO_CONFIG:
                platform.values[os_] = platform.Value(os_, default=(os_ == 'android'))

        with sdk2.parameters.Output:
            artifacts = sdk2.parameters.Resource("Artifacts")
            deb_pkg_json = sdk2.parameters.Resource('Debian package second pkg.json')
            debian_package = sdk2.parameters.Resource('Debian package',
                description='A .tar.gz containing a .deb',
                )

    def _set_up(self):
        super(MapsMobileMakeArtifacts, self)._set_up()
        self._config = _OS_TO_CONFIG[self.Parameters.platform]
        self._bundle_aar_path = (
                'maps/mobile/apps/test_app/bundle/aar'
                if _OLD_BUNDLE_AAR_REGEX.search(self.Parameters.release_version)
                else 'maps/mobile/bundle/aar'
                )
        self._is_android_local_so_from_aar = (
                self.Parameters.platform == 'android'
                and self.Parameters.is_navi
                and is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_LOCAL_SO_FROM_AAR)
                )
        self._is_separately_built_android_local_so = (
                self.Parameters.platform == 'android'
                and self.Parameters.is_navi
                and not is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_LOCAL_SO_FROM_AAR)
                )
        self._aar_build_key_prefix = (
                'build_aar'
                if self._is_separately_built_android_local_so
                else 'build'
                )

    def _postprocess(self):
        postprocess = []
        if self.Parameters.platform == 'android':
            postprocess += self._postprocess_android_packager()
        if self.Parameters.is_navi:
            postprocess += self._postprocess_mv_local_headers()
        return postprocess

    def _build_with_ya_package(self):
        subtask_type = self._config.ya_package_tarball_task_type
        with self.memoize_stage['run_{}'.format(subtask_type)]:
            subtask_parameters = {
                build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build.parameters.ForceVCSInfoUpdate.name: True,
                build.parameters.UseArcadiaApiFuse.name: True,
                build.parameters.UseArcInsteadOfArcadiaApi.name: True,
                build.YaPackage.AdhocPackagesParameter.name: [self._pkg_json],
                build.YaPackage.PackageTypeParameter.name: 'tarball',
                build.YaPackage.SaveBuildOutputParameter.name: True,
                'kill_timeout': 6 * 60 * 60,
                'ya_timeout': 6 * 60 * 60
            }
            subtask_parameters.update(yt_store_parameters())
            SubtaskRunner(self).run(subtask_type, subtask_parameters)
        with self.memoize_stage['handle_{}_result'.format(subtask_type)]:
            subtask = SubtaskRunner(self).find_task(subtask_type)
            SubtaskRunner.require_successful(subtask)
            resource = get_ya_package_resource(subtask)
            if self.Parameters.is_navi:
                apply_navi_export_parameters(resource)
            self.Parameters.artifacts = resource

    def _provides_conflicts_replaces(self):
        provides_conflicts_replaces = []
        if self._is_android_local_so_from_aar:
            provides_conflicts_replaces += [
                "yandex-mapsmobi-android-libcxx",
            ]
        if self.Parameters.platform in ['linux', 'darwin']:
            provides_conflicts_replaces += [
                "yandex-mapsmobi-contrib-qt-{}".format(self.Parameters.platform),
            ]
        return provides_conflicts_replaces

    def _building_pkg_json_build(self):
        build = {}
        if self.Parameters.platform in ['darwin', 'linux'] or self._is_separately_built_android_local_so:
            build.update({
                "build_{}".format(arch): self._build_item(arch=arch,
                                                          target='maps/mobile/bundle/dynamic',
                                                          )
                for arch in self._architectures
            })
        build_tools_targets = []
        if self.Parameters.platform == 'android':
            for arch in self._architectures:
                platform_flags = [
                    ('OS_SDK', 'ubuntu-16'),
                ]
                if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_AAR_NO_STRIP):
                    platform_flags += [('AAR_NO_STRIP', 'yes')]
                else:
                    platform_flags += [('NO_DEBUGINFO', 'yes')]
                if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_REDUCED_DEBUGINFO):
                    platform_flags += [('REDUCED_DEBUGINFO', 'yes')]

                build["{}_{}".format(self._aar_build_key_prefix, arch)] = self._build_item(
                        arch=arch,
                        target=self._bundle_aar_path,
                        platform_flags=platform_flags)
            build_tools_targets.append("maps/mobile/tools/android-packager")
        if self.Parameters.is_navi:
            build_tools_targets.append("maps/mobile/tools/ya-package-mv")
        if build_tools_targets:
            build["build_tools"] = {
                "targets": build_tools_targets
            }
        return build

    def _navi_build_flags(self):
        return [
            ('MAPS_MOBILE_EXPORT_CPP_API', 'yes'),
        ]

    def _building_pkg_json_data(self):
        data = super(MapsMobileMakeArtifacts, self)._building_pkg_json_data()
        if self.Parameters.platform in ['darwin', 'linux'] or self._is_separately_built_android_local_so:
            data += self._local_libraries_data()
        if self.Parameters.platform == 'android':
            data += self._android_bundle_data()
        if self._is_android_local_so_from_aar:
            data += self._android_libcxx_headers_data()
        return data

    def _android_bundle_data(self):
        # An arbitrary architecture. We need only one copy of xmls,
        # jars and javadoc from any of the build aars.
        xml_jar_arch = self._architectures[0]
        return [
            {
                "source": {
                    "type": "BUILD_OUTPUT",
                    "build_key": "{}_{}".format(self._aar_build_key_prefix, xml_jar_arch),
                    "path": self._bundle_aar_path,
                    "files": ["*.jar", "*.xml"] + (["*.tar.gz"] if self.Parameters.generate_doc else [])
                },
                "destination": {
                    "path": "/bundle/"
                }
            }
        ] + [
            {
                "source": {
                    "type": "BUILD_OUTPUT",
                    "build_key": "{}_{}".format(self._aar_build_key_prefix, arch),
                    "path": os.path.join(self._bundle_aar_path, "maps.mobile.aar"),
                },
                "destination": {
                    "path": ("/bundle/maps.mobile.symbols.aar.{toolchain}"
                             .format(toolchain=self._toolchain(arch)))
                }
            }
            for arch in self._architectures
        ]

    def _android_maps_mobile_json_data(self):
        return [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": "maps/mobile/bundle/maps-mobile.json",
                       },
                       "destination": {
                           "path": "/bundle/android/maps-mobile.json"
                       }
                   }
               ]

    def _qt_data(self):
        return [
                   {
                       "source": {
                           "type": "SANDBOX_RESOURCE",
                           "id": self._config.qt_resource,
                           "untar": True,
                           "files": ["*"],
                           "symlinks": True,
                       },
                       "destination": {
                           "path": self._local_dir(arch)
                       }
                   }
                   for arch in self._architectures
                   if self._config.qt_resource
               ]

    def _sandbox_id_for_typeinfo(self):
        return 2527848662 if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_NDK23) else 1421157295

    def _android_libcxx_headers_data(self):
        # Sandbox Resource ID duplicates
        # https://a.yandex-team.ru/arc/trunk/arcadia/build/platform/mapkit/ya.make?rev=7469463#L9
        return [
                    {
                        "source": {
                            "type": "SANDBOX_RESOURCE",
                            "id": self._sandbox_id_for_typeinfo(),
                            "files": ["__config", "typeinfo"],
                            "untar": True,
                        },
                        "destination": {
                            "path": self._local_dir(arch, "include/android-libcxx"),
                        }
                    }
                    for arch in self._architectures
                ]

    def _arcadia_linux_libcxx_data(self):
        arch = self._architectures[0]
        return [
                    {
                        "source": {
                            "type": "ARCADIA",
                            "path": "contrib/libs/cxxsupp/libcxx/include",
                            "files": ["*"]
                        },
                        "destination": {
                            "path": self._local_dir(arch, "include")
                        }
                    },
                    {
                        "source": {
                            "type": "ARCADIA",
                            "path": "contrib/libs/cxxsupp/libcxxrt",
                            "files": ["*.h"]
                        },
                        "destination": {
                            "path": self._local_dir(arch, "include/contrib/libs/cxxsupp/libcxxrt")
                        }
                    },
                    {
                        "source": {
                            "type": "ARCADIA",
                            "path": "contrib/libs/libunwind",
                            "files": ["*.h"]
                        },
                        "destination": {
                            "path": self._local_dir(arch, "include/contrib/libs/libunwind")
                        }
                    }
                ]

    def _local_libraries_data(self):
        return [{
                    "source": {
                        "type": "BUILD_OUTPUT",
                        "path": 'maps/mobile/bundle/dynamic',
                        "files": ['*.{}'.format(ext) for ext in self._config.dynamic_library_extensions],
                        "build_key": "build_{arch}".format(arch=arch),
                    },
                    "destination": {
                        "path": self._local_dir(arch, 'lib'),
                    }
                }
                for arch in self._architectures]

    def _postprocess_android_packager(self):
        return [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "build_key": "build_tools",
                           "path": "maps/mobile/tools/android-packager/android-packager"
                       },
                       "arguments": self._android_packager_arguments()
                   }
               ]

    def _android_packager_arguments(self):
        arguments = [
                        "--input-aar-name-prefix", 'maps.mobile.symbols.aar',
                        "--output-aar-name", 'maps.mobile.symbols.aar',
                        "--stripped-output-aar-name", 'maps.mobile.aar',
                        "--aar-dir", "./bundle",
                    ]
        if self._is_android_local_so_from_aar:
            arguments += [
                             "--local-dir", "./local",
                         ]
        return arguments

    def _deb_pkg_json_data(self):
        data = []
        data += self._deb_pkg_json_artifacts_data()
        if self.Parameters.platform == 'android':
            data += self._android_maps_mobile_json_data()
        if self.Parameters.platform in ['darwin', 'linux']:
            data += self._qt_data()
        if self.Parameters.platform == 'linux':
            data += self._arcadia_linux_libcxx_data()
        return data

    def _deb_pkg_json_artifacts_data(self):
        return [
            {
                "source": {
                    "type": "SANDBOX_RESOURCE",
                    "id": self.Parameters.artifacts.id,
                    "untar": True,
                    "symlinks": True,
                },
                "destination": {
                    "path": "/",
                }
            },
        ]

    def _publish_javadoc(self):
        output_path = os.path.abspath(
                'YandexMapsMobile-{}-javadoc'
                .format(self.Parameters.release_version))
        unzip_path = os.path.abspath('untarred_artifacts')
        if os.path.isdir(output_path):
            raise TaskError('Failed to create a directory for Javadoc: {} '
                              'already exists!'.format(output_path))
        os.mkdir(output_path)
        os.mkdir(unzip_path)

        with tarfile.open(
                str(sdk2.ResourceData(self.Parameters.artifacts).path)) as artifacts:
            artifacts.extract(
                    str(os.path.join('bundle', 'maps.mobile-javadoc.tar.gz')),
                    path=unzip_path)

        with tarfile.open(
                str(os.path.join(unzip_path, 'bundle', 'maps.mobile-javadoc.tar.gz'))) as javadoc:
            javadoc.extractall(path=output_path)

        self.Parameters.documentation = MapsMobileJavadocDocumentation(
            self,
            'Raw Javadoc docummentation',
            output_path,
            ttl=_JAVADOC_DOCS_TTL,
            backup_task=True)
        sdk2.ResourceData(self.Parameters.documentation).ready()
