import os
import yaml

from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as arcadia_sdk

from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
        import validate_has_exact_svn_revision


_TOOLCHAINS_PATH = os.path.join('maps', 'mobile', '.build_config.yaml')


class MapsMobileMakeArtifactsBase(sdk2.Task):
    ''' Base Task for common code used in building artifacts for all platforms. '''

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Arcadia URL with a required revision:',
                description='An example: arcadia:/arc/trunk/arcadia@6150000',
                required=True)
        release_version = sdk2.parameters.String('Release version', required=True)
        libraries = sdk2.parameters.List('Libraries')
        generate_doc = sdk2.parameters.Bool(
                'Generate documentation',
                description='Build Javadoc documentation',
                default=False)

        with sdk2.parameters.Output:
            pkg_json = sdk2.parameters.Resource('Generated pkg.json')

    def _set_up(self):
        validate_has_exact_svn_revision(self.Parameters.arcadia_url)
        self._read_toolchains()
        self._libraries = ['runtime'] + self.Parameters.libraries

    def _read_toolchains(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            with open(os.path.join(arcadia, _TOOLCHAINS_PATH)) as f:
                self._toolchains_config = yaml.safe_load(f)

    def _meta(self):
        meta = {
                   "name": self._package_name(),
                   "maintainer": "Yandex LLC <maps-api@support.yandex.ru>",
                   "description": ("An ad-hoc maps-mobile package"
                                   " generated by {task_type} sandbox task with ID: {task_id}."
                                   " Manual task ID: {root_task_id}"
                                   .format(task_type=self.type,
                                           task_id=self.id,
                                           root_task_id=self._root_task_id())),
                   "version": self.Parameters.release_version,
               }
        provides_conflicts_replaces = self._provides_conflicts_replaces()
        if provides_conflicts_replaces:
            meta["provides"] = provides_conflicts_replaces
            meta["conflicts"] = provides_conflicts_replaces
            meta["replaces"] = provides_conflicts_replaces
        depends = self._depends()
        if depends:
            meta["depends"] = depends
        return meta

    def _package_name(self):
        platform_suffix = ""
        if self._package_name_platform():
            platform_suffix = "-{}".format(self._package_name_platform())
        return "yandex-mapsmobi-maps-mobile-bundle{}".format(platform_suffix)

    def _package_name_platform(self):
        return None

    def _provides_conflicts_replaces(self):
        return []

    def _depends(self):
        return []

    def _build_item(self, arch, target, platform_flags=[]):
        return {
                   "targets": [
                       target
                   ],
                   "target-platforms": [
                       self._toolchain(arch)
                   ],
                   "add-result": [
                       ".h"
                   ],
                   "flags": [{
                                 "name": name,
                                 "value": value,
                             }
                             for name, value in (
                                 self._common_build_flags()
                                 + platform_flags
                                 + (self._navi_build_flags() if self.Parameters.is_navi else [])
                             )],
               }

    def _common_build_flags(self):
        return [
                   ('MAPSMOBI_BUILD_TARGET', 'yes'),
                   ('SANDBOX_TASK_ID', self._root_task_id()),
                   ('AAR_GENERATE_DOC', 'yes' if self.Parameters.generate_doc else 'no'),
               ] + [
                   ('WITH_' + lib.upper(), 'yes')
                   for lib in self.Parameters.libraries
               ]

    def _navi_build_flags(self):
        return []

    def _toolchain(self, arch):
        return '{prefix}-{platform}-{arch}'.format(
            prefix=self._toolchains_config[self.Parameters.platform]['toolchain_prefix'],
            platform='iossim' if arch == 'm1sim' else self.Parameters.platform,
            arch='arm64' if arch == 'm1sim' else arch,
        )

    def _postprocess(self):
        return None

    def _directories(self, take_dir):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            return [
                os.path.relpath(full_dir, os.path.abspath(arcadia))
                for library in self._libraries
                for full_dir, _, _ in os.walk(self._arcadia_library_path(arcadia, library))
                if full_dir.endswith(os.sep + take_dir)
            ]

    def _arcadia_library_path(self, arcadia, library):
        return os.path.join(arcadia, 'maps', 'mobile', 'libs', library)

    def _root_task_id(self):
        task = self
        while task.parent:
            task = task.parent
        return str(task.id)
