import json
import os
import re

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import build
import sandbox.projects.common.arcadia.sdk as arcadia_sdk

from sandbox.projects.maps.mobile.MapsMobileMakeArtifactsBase \
        import MapsMobileMakeArtifactsBase
from sandbox.projects.maps.mobile.MapsMobileResources \
        import MapsMobileExportArtifactsPkgJson
from sandbox.projects.maps.mobile.utils.resource_helpers import (
    make_resource,
    apply_navi_export_parameters,
    get_ya_package_resource,
    )
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date
from sandbox.projects.maps.mobile.utils.yt_store_parameters import yt_store_parameters


# An arbitrary architecture. We need only one copy of protobuf headers.
_PB_H_ARCH = 'x86_64'
_PKG_JSON_TTL = 30
# An arbitrary platform. The artifacts in dev bundle should be
# platform-independent
_PLATFORM = 'linux'
_YA_PACKAGE_SUBTASK_TYPE = 'MAPS_MOBILE_YA_PACKAGE_LINUX_XENIAL'
# Remove these consts and their uses when support for corresponding older branches is dropped.
_FIRST_BRANCH_WITH_PROTOC_AND_WITH_INC_IN_PROTOBUF_STD = '2020062519'
_FIRST_BRANCH_WITH_RESTRICTED_GOOGLETEST = '2020071918'
_AUGUST_TWO_BRANCH_WITHOUT_GOOGLETEST = '2020082111'
_FIRST_BRANCH_WITHOUT_GOOGLETEST = '20200916'


class MapsMobileMakeDevBundle(MapsMobileMakeArtifactsBase):
    ''' Task for packaging artifacts. '''

    class Parameters(MapsMobileMakeArtifactsBase.Parameters):
        # An arbitrary platform. The artifacts in dev bundle should be
        # platform-independent
        platform = _PLATFORM
        is_navi = True
        is_upload = sdk2.parameters.Bool('Upload to dist?')

        with sdk2.parameters.Output:
            debian_package = sdk2.parameters.Resource("Debian package")

    def on_execute(self):
        self._set_up()
        with self.memoize_stage.make_pkg_json:
            self._pkg_json = self._make_pkg_json()
            self.Parameters.pkg_json = make_resource(
                    self,
                    MapsMobileExportArtifactsPkgJson,
                    'pkg.json for the task with id {}'.format(self.id),
                    'pkg.json',
                    self._pkg_json,
                    ttl=_PKG_JSON_TTL,
                    )
        self._run_ya_package_debian()
        self._run_ya_package_tarball()

    def _run_ya_package_debian(self):
        subtask_type = _YA_PACKAGE_SUBTASK_TYPE
        with self.memoize_stage.run_subtask_debian:
            subtask_parameters = {
                build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build.parameters.UseArcadiaApiFuse.name: True,
                build.parameters.UseArcInsteadOfArcadiaApi.name: True,
                build.YaPackage.AdhocPackagesParameter.name: [self._pkg_json],
                build.YaPackage.PackageTypeParameter.name: 'debian',
                build.YaPackage.SaveBuildOutputParameter.name: True,
                'kill_timeout': 6 * 60 * 60,
                build.YaPackage.PublishPackageParameter.name: self.Parameters.is_upload,
            }
            subtask_parameters.update(yt_store_parameters())
            if self.Parameters.is_upload:
                subtask_parameters[build.YaPackage.PublishToParameter.name] = 'mapsmobi-common'
                subtask_parameters[build.YaPackage.ArchitectureAllParameter.name] = True
            SubtaskRunner(self).run(subtask_type, subtask_parameters)
        with self.memoize_stage.handle_result_debian:
            subtask = SubtaskRunner(self).find_task(subtask_type)
            SubtaskRunner.require_successful(subtask)
            resource = get_ya_package_resource(subtask)
            self.Parameters.debian_package = resource

    def _run_ya_package_tarball(self):
        with open(str(sdk2.ResourceData(self.Parameters.pkg_json).path), 'r') as f:
            self._pkg_json = f.read()
        subtask_parameters = {
            build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
            build.parameters.UseArcadiaApiFuse.name: True,
            build.parameters.UseArcInsteadOfArcadiaApi.name: True,
            build.YaPackage.AdhocPackagesParameter.name: [self._pkg_json],
            build.YaPackage.PackageTypeParameter.name: 'tarball',
            build.YaPackage.SaveBuildOutputParameter.name: True,
            'kill_timeout': 6 * 60 * 60,
        }
        SubtaskRunner(self).run_and_handle_result(
            _YA_PACKAGE_SUBTASK_TYPE, subtask_parameters, self._handle_tarball, tag='tarball')

    def _handle_tarball(self, subtask):
        resource = get_ya_package_resource(subtask)
        apply_navi_export_parameters(resource)
        resource.host_platform = _PLATFORM


    def _make_pkg_json(self):
        pkg_json_object = {
            "meta": self._meta(),
            "build": self._pkg_json_build(),
            "data": self._pkg_json_data(),
        }
        postprocess = self._postprocess()
        if postprocess:
            pkg_json_object["postprocess"] = postprocess
        return json.dumps(pkg_json_object, indent=4)

    def _depends(self):
        dependencies = [
            "yandex-mapsmobi-tools-idl-app (= {})".format(self.Parameters.release_version),
        ]
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_PROTOC_AND_WITH_INC_IN_PROTOBUF_STD):
            dependencies += [
                "yandex-mapsmobi-contrib-protobuf-common (= {})".format(self.Parameters.release_version),
            ]
        return dependencies

    def _pkg_json_build(self):
        return {
            "build_{}".format(_PB_H_ARCH): self._build_item(
                    arch=_PB_H_ARCH,
                    target='maps/mobile/bundle/dynamic'),
        }

    def _pkg_json_data(self):
        protobuf_path = (
            "contrib/libs/protobuf_std"
            if re.search('/20200([34]|506)', self.Parameters.arcadia_url)
            else "contrib/libs/protobuf_std/src"
        )
        protobuf_files = (
            ["*.h", "*.inc"]
            if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_PROTOC_AND_WITH_INC_IN_PROTOBUF_STD)
            else ["*.h"]
        )
        gmock_path = (
            "contrib/restricted/googletest/googlemock/include"
            if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_RESTRICTED_GOOGLETEST)
            else "contrib/libs/gmock/include"
        )
        gtest_path = (
            "contrib/restricted/googletest/googletest/include"
            if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_RESTRICTED_GOOGLETEST)
            else "contrib/libs/gtest/include"
        )
        return (
            self._pb_h_data()
            + self._idls_and_frameworks_data()
            + self._contrib_headers_data([
                    "contrib/libs/libpng",
                    "contrib/libs/rapidjson/include",
                    "contrib/libs/zlib",
                    "contrib/libs/zstd",
            ], files=["*.h"])
            + (
                [] if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITHOUT_GOOGLETEST)
                    and not self.Parameters.release_version.startswith(_AUGUST_TWO_BRANCH_WITHOUT_GOOGLETEST)
                else self._contrib_headers_data([gmock_path, gtest_path], files=["*.h"])
            )
            + self._contrib_headers_data([protobuf_path], files=protobuf_files)
            + self._contrib_headers_data(["contrib/libs/eigen"], files=["Eigen/*"],
                                         used_by_full_arcadia_path=True)
            + self._contrib_headers_data(["contrib/restricted/boost"], files=["*.h", "*.hpp", "*.ipp"])
            + self._contrib_headers_data(["util/system"], files=["compiler.h"],
                                         used_by_full_arcadia_path=True)
        )

    def _pb_h_data(self):
        return [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "build_key": "build_{}".format(_PB_H_ARCH),
                           "path": "maps/doc/proto",
                           "files": ["*.pb.h"],
                       },
                       "destination": {
                           "path": "/include/",
                       }
                   }
               ]

    def _idls_and_frameworks_data(self):
        self._idl_dirs = self._directories(take_dir='idl')
        return [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": idl_dir,
                           "files": [
                               "*.idl",
                           ],
                       },
                       "destination": {
                           "path": "/share/idl/",
                       }
                   }
                   for idl_dir in self._idl_dirs
               ] + [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": "maps/mobile/libs/idl_frameworks",
                           "files": [
                               "{}.framework".format(idl_framework_name)
                               for idl_framework_name in self._idl_frameworks()
                           ],
                       },
                       "destination": {
                           "path": "/share/idl_frameworks/",
                       }
                   }
               ]

    def _contrib_headers_data(self, paths, files, used_by_full_arcadia_path=False):
        return [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": path,
                           "files": files,
                       },
                       "destination": {
                           "path": "/include/{}".format(path + "/" if used_by_full_arcadia_path else ""),
                       }
                   }
                   for path in paths
               ]

    def _idl_frameworks(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            return {
                framework_name
                for idl_dir in self._idl_dirs
                for framework_name in os.listdir(os.path.join(arcadia, idl_dir))
                if os.path.isdir(os.path.join(arcadia, idl_dir, framework_name))
            }
