import json
import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import build

from sandbox.projects.maps.mobile.utils.resource_helpers import make_resource
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner

from sandbox.projects.maps.mobile.MapsMobileMakeArtifactsBase \
        import MapsMobileMakeArtifactsBase
from sandbox.projects.maps.mobile.MapsMobileResources \
        import MapsMobileExportArtifactsPkgJson, MapsMobileDebianPackagePkgJson


_PKG_JSON_TTL = 30
_LIBS_WITHOUT_IDL_HEADERS = {'auth'}


class MapsMobileMakePlatformArtifactsBase(MapsMobileMakeArtifactsBase):
    ''' Base Task for common code used in building artifacts for all platforms. '''

    class Parameters(MapsMobileMakeArtifactsBase.Parameters):
        is_navi = sdk2.parameters.Bool('Build for Navi?',
                                       description='Build for Navi: debian package with bundle/ and local/.'
                                                   ' For iOS also mangled_symbols. \n'
                                                   'Otherwise: just a framework zip.',
                                       default=False)
        with is_navi.value[True]:
            is_upload = sdk2.parameters.Bool('Upload to dist?')

        with sdk2.parameters.Output:
            documentation = sdk2.parameters.Resource('Raw Javadoc documentation')

    def on_execute(self):
        self._set_up()
        with self.memoize_stage.generate_building_pkg_json:
            self._pkg_json = self._generate_building_pkg_json()
            self._make_building_pkg_json_resource()
        self._build_with_ya_package()
        if self.Parameters.is_navi:
            with self.memoize_stage.generate_deb_pkg_json:
                self._deb_pkg_json = self._generate_deb_pkg_json()
                self._make_deb_pkg_json_resource()
            self._ya_package_debian()
        if self.Parameters.platform == 'android' and self.Parameters.generate_doc:
            self._publish_javadoc()

    def _set_up(self):
        super(MapsMobileMakePlatformArtifactsBase, self)._set_up()
        self._idl_header_libraries = set(self._libraries) - _LIBS_WITHOUT_IDL_HEADERS
        self._architectures = list(
            self._toolchains_config[self.Parameters.platform]['architecture_to_local_dir'].keys())

    def _package_name_platform(self):
        return self.Parameters.platform

    def _generate_building_pkg_json(self):
        pkg_json_object = {
            "meta": self._meta(),
            "build": self._building_pkg_json_build(),
            "data": self._building_pkg_json_data(),
        }
        postprocess = self._postprocess()
        if postprocess:
            pkg_json_object["postprocess"] = postprocess
        return json.dumps(pkg_json_object, indent=4)

    def _generate_deb_pkg_json(self):
        pkg_json_object = {
            "meta": self._meta(),
            "data": self._deb_pkg_json_data()
        }
        return json.dumps(pkg_json_object, indent=4)

    def _local_dir(self, arch, subdir=None):
        local_dir_path_dirs = [
            os.sep + 'local',
            '{platform}.{arch_suffix}'.format(
                platform='iossim' if arch == 'm1sim' else self.Parameters.platform,
                arch_suffix=self._toolchains_config[self.Parameters.platform]['architecture_to_local_dir'][arch],
                )
        ]
        if subdir:
            local_dir_path_dirs.append(subdir)
        return os.path.join(*local_dir_path_dirs) + os.sep

    def _building_pkg_json_data(self):
        data = []
        if self.Parameters.is_navi:
            data += self._local_headers_data()
        return data

    def _local_headers_data(self):
        return [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": source_path,
                           "files": ["*.h"],
                       },
                       "destination": {
                           "path": self._local_dir(arch, 'include'),
                       }
                   }
                   for arch in self._architectures
                   for source_path in self._directories('include')
               ] + [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "path": "maps/mobile/libs/{lib}".format(lib=lib),
                           "files": [
                               "*/idl/include/*.h"
                           ],
                           "build_key": "build_{arch}".format(arch=arch),
                       },
                       "destination": {
                           "path": '/temp_local_headers/{arch}/'.format(arch=arch),
                       }
                   }
                   for arch in self._architectures
                   for lib in self._idl_header_libraries
               ]

    def _postprocess_mv_local_headers(self):
        return [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "build_key": "build_tools",
                           "path": "maps/mobile/tools/ya-package-mv/ya-package-mv",
                       },
                       "arguments": [
                           "--source", "./temp_local_headers/{arch}/*/idl/include".format(arch=arch),
                           "--destination", ('.' + self._local_dir(arch, "include")),
                       ],
                   }
                   for arch in self._architectures
               ]

    def _make_building_pkg_json_resource(self):
        self.Parameters.pkg_json = make_resource(
                self,
                MapsMobileExportArtifactsPkgJson,
                'First pkg.json for building' if self.Parameters.is_navi else 'pkg.json',
                'building_pkg.json',
                self._pkg_json,
                ttl=_PKG_JSON_TTL,
                )

    def _make_deb_pkg_json_resource(self):
        self.Parameters.deb_pkg_json = make_resource(
                self,
                MapsMobileDebianPackagePkgJson,
                'Second pkg.json for repackaging to a .deb',
                'deb_pkg.json',
                self._deb_pkg_json,
                ttl=_PKG_JSON_TTL,
                )

    def _ya_package_debian(self):
        with self.memoize_stage.run_ya_package_debian:
            subtask_parameters = {
                build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build.parameters.UseArcadiaApiFuse.name: True,
                build.parameters.UseArcInsteadOfArcadiaApi.name: True,
                build.YaPackage.AdhocPackagesParameter.name: [self._deb_pkg_json],
                build.YaPackage.PackageTypeParameter.name: 'debian',
                build.YaPackage.SaveBuildOutputParameter.name: True,
                build.YaPackage.PublishPackageParameter.name: self.Parameters.is_upload,
            }
            if self.Parameters.is_upload:
                subtask_parameters[build.YaPackage.PublishToParameter.name] = 'mapsmobi-common'
                subtask_parameters[build.YaPackage.ArchitectureAllParameter.name] = True
            SubtaskRunner(self).run('MAPS_MOBILE_YA_PACKAGE_DEB', subtask_parameters)
        with self.memoize_stage.handle_ya_package_debian_result:
            subtask = SubtaskRunner(self).find_task('MAPS_MOBILE_YA_PACKAGE_DEB')
            SubtaskRunner.require_successful(subtask)
            resource = sdk2.Resource['YA_PACKAGE'].find(task=subtask).first()
            if not resource:
                raise TaskError('Could not find YA_PACKAGE resource')
            self.Parameters.debian_package = resource
