import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileFrameworkZip
from sandbox.projects.maps.mobile.utils.resource_helpers import (
    extract_resource,
    make_resource,
    apply_navi_export_parameters,
    )
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
        import validate_has_exact_svn_revision
from sandbox.projects.maps.mobile.utils.zip_helpers import pack_into_zip_file


_FRAMEWORK_ZIP_TTL = 2
_BUILD_STATIC_FRAMEWORK_TASK_TYPE = "MAPS_MOBILE_BUILD_IOS_FAT_PROJECT"
_STATIC_FRAMEWORK_RESOURCE_TYPE = "MAPS_MOBILE_IOS_FRAMEWORK"
_BUILD_DYNAMIC_FRAMEWORK_TASK_TYPE = "MAPS_MOBILE_BUILD_IOS_XCFRAMEWORK"
_DYNAMIC_FRAMEWORK_RESOURCE_TYPE = "MAPS_MOBILE_IOS_XCFRAMEWORK"
_BUILD_RESOURCE_BUNDLE_TASK_TYPE = "MAPS_MOBILE_BUILD_IOS_RESOURCE_BUNDLE"
_RESOURCE_BUNDLE_RESOURCE_TYPE = "MAPS_MOBILE_IOS_RESOURCE_BUNDLE"

class MapsMobileMakePlatformIosFramework(sdk2.Task):
    ''' Task for making platform framework zip for distribution. '''

    class Parameters(sdk2.Task.Parameters):

        arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Arcadia URL with a required revision:',
                description='An example: arcadia:/arc/trunk/arcadia@6150000',
                required=True)
        release_version = sdk2.parameters.String('Release version', required=True)
        maps_mobile_bundle_resource = sdk2.parameters.Resource('Maps mobile bundle tar.gz', required=True)
        framework_project = sdk2.parameters.String('Framework project name', required=True)
        bundle_project = sdk2.parameters.String('Resource bundle project name', required=True)
        is_dynamic = sdk2.parameters.Bool('Is dynamic framework', required=True)
        project_src_path = sdk2.parameters.String('Path to project sources', required=True)
        is_navi = sdk2.parameters.Bool('Build for Navi?',
                                       description='Build for Navi: special parameters on build task resources.',
                                       default=False)
        with sdk2.parameters.Output:
            framework_zip = sdk2.parameters.Resource('Framework zip')

    def _build_framework(self):
        subtask_type = (_BUILD_DYNAMIC_FRAMEWORK_TASK_TYPE
            if self.Parameters.is_dynamic
            else _BUILD_STATIC_FRAMEWORK_TASK_TYPE)
        framework_scheme = self.Parameters.framework_project
        if self.Parameters.is_dynamic:
            framework_scheme += 'Dynamic'
        else:
            framework_scheme += 'Static'
        arcadia_url_tokens = self.Parameters.arcadia_url.split('@')
        subtask_parameters = {
            'arcadia_url': arcadia_url_tokens[0],
            'arcadia_revision': arcadia_url_tokens[1],
            'build_variant': 'release',
            'prebuilt_bundle': True,
            'bundle_resource': self.Parameters.maps_mobile_bundle_resource,
            'project': self.Parameters.framework_project,
            'scheme': framework_scheme,
            'project_src_path': self.Parameters.project_src_path
        }
        if not self.Parameters.is_dynamic:
            subtask_parameters['project_type'] = 'Framework'
        subtask_requirements = {
            'disk_space': 20 * 1024  # MiB
        }
        SubtaskRunner(self).run_and_handle_result(
            subtask_type,
            subtask_parameters,
            subtask_requirements=subtask_requirements,
            result_callback=self._handle_framework_build_result,
            tag='framework',
            require_successful=True,
            )

    def _handle_framework_build_result(self, subtask):
        framework_resource_type = (_DYNAMIC_FRAMEWORK_RESOURCE_TYPE
            if self.Parameters.is_dynamic
            else _STATIC_FRAMEWORK_RESOURCE_TYPE)
        resource = sdk2.Resource[framework_resource_type].find(task=subtask).first()
        if not resource:
            raise TaskError('Could not find framework resource')
        if self.Parameters.is_navi:
            apply_navi_export_parameters(resource)
        self.Context.framework_artifacts = resource.id



    def _build_bundle(self):
        subtask_type = _BUILD_RESOURCE_BUNDLE_TASK_TYPE
        arcadia_url_tokens = self.Parameters.arcadia_url.split('@')
        subtask_parameters = {
            'arcadia_url': arcadia_url_tokens[0],
            'arcadia_revision': arcadia_url_tokens[1],
            'build_variant': 'release',
            'prebuilt_bundle': True,
            'bundle_resource': self.Parameters.maps_mobile_bundle_resource,
            'project': self.Parameters.bundle_project,
            'scheme': self.Parameters.bundle_project,
            'project_src_path': self.Parameters.project_src_path
        }
        subtask_requirements = {
            'disk_space': 20 * 1024  # MiB
        }
        SubtaskRunner(self).run_and_handle_result(
            subtask_type, subtask_parameters,
            subtask_requirements=subtask_requirements,
            result_callback=self._handle_bundle_build_result,
            tag='bundle',
            require_successful=True,
            )

    def _handle_bundle_build_result(self, subtask):
        resource = sdk2.Resource[_RESOURCE_BUNDLE_RESOURCE_TYPE].find(task=subtask).first()
        if not resource:
            raise TaskError('Could not find resource bundle resource')
        if self.Parameters.is_navi:
            apply_navi_export_parameters(resource)
        self.Context.bundle_artifacts = resource.id

    def _set_up(self):
        validate_has_exact_svn_revision(self.Parameters.arcadia_url)

    def _make_framework_zip(self):
        tempdir = os.path.abspath('artifacts')
        extract_resource(sdk2.Resource[self.Context.framework_artifacts], tempdir)
        extract_resource(sdk2.Resource[self.Context.bundle_artifacts], tempdir)
        tempdir_paths = [os.path.join(tempdir, '{}.bundle'.format(self.Parameters.bundle_project))]
        if self.Parameters.is_dynamic:
            tempdir_paths += [os.path.join(tempdir, '{}.xcframework'.format(self.Parameters.framework_project))]
        else:
            tempdir_paths += [os.path.join(tempdir, '{}.framework'.format(self.Parameters.framework_project))]
        framework_zip_path = '{}-{}.framework.zip'.format(
                self.Parameters.framework_project, self.Parameters.release_version)
        zip_bytes = pack_into_zip_file(tempdir, tempdir_paths)
        self.Parameters.framework_zip = make_resource(
                self,
                MapsMobileFrameworkZip,
                description='Framework zip',
                path=framework_zip_path,
                content=zip_bytes,
                ttl=_FRAMEWORK_ZIP_TTL,
                )

    def on_execute(self):
        self._set_up()
        self._build_framework()
        self._build_bundle()
        with self.memoize_stage.make_framework_zip:
            self._make_framework_zip()
