import json
import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import build

from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
        import validate_has_exact_svn_revision
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.maps.mobile.utils.yt_store_parameters import yt_store_parameters
from sandbox.projects.maps.mobile.utils.resource_helpers import (
    apply_navi_export_parameters,
    get_ya_package_resource,
    )


_PACKAGE_LINUX_TASK_TYPE = 'MAPS_MOBILE_YA_PACKAGE_LINUX_XENIAL'
_DARWIN_BUILD_TASK_TYPE = 'MAPS_MOBILE_YA_PACKAGE_DARWIN'
_DARWIN_DEBIAN_TASK_TYPE = 'MAPS_MOBILE_YA_PACKAGE_DEB'


class MapsMobileMakeTools(sdk2.Task):
    ''' Task for packaging artifacts. '''

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Arcadia URL with a required revision:',
                description='An example: arcadia:/arc/trunk/arcadia@6150000',
                required=True)
        building_pkg_json_path = sdk2.parameters.String(
                'pkg.json path',
                description='An example: tools/idl/package/pkg.json',
                required=True)
        release_version = sdk2.parameters.String('Release version', required=True)
        is_upload = sdk2.parameters.Bool('Upload to dist?')

    def on_execute(self):
        validate_has_exact_svn_revision(self.Parameters.arcadia_url)
        self._ya_package_linux_tar_gz()
        self._ya_package_linux_debian()
        self._build_for_darwin()
        self._publish_darwin_debian()

    def _ya_package_linux_tar_gz(self):
        subtask_type = _PACKAGE_LINUX_TASK_TYPE
        subtask_parameters = self._build_parameters()
        subtask_parameters.update({
            build.YaPackage.PackageTypeParameter.name: 'tarball',
        })
        SubtaskRunner(self).run_and_handle_result(
                subtask_type, subtask_parameters, self._handle_linux_tarball, tag='ya_package_linux_tar_gz')

    def _ya_package_linux_debian(self):
        subtask_type = _PACKAGE_LINUX_TASK_TYPE
        subtask_parameters = self._build_parameters()
        subtask_parameters.update(self._debian_parameters('linux'))
        SubtaskRunner(self).run_and_handle_result(subtask_type, subtask_parameters, tag='ya_package_linux_debian')

    def _build_for_darwin(self):
        subtask_type = _DARWIN_BUILD_TASK_TYPE
        subtask_parameters = self._build_parameters()
        subtask_parameters.update({
            build.YaPackage.PackageTypeParameter.name: 'tarball',
            build.YaPackage.SaveBuildOutputParameter.name: True,
        })
        SubtaskRunner(self).run_and_handle_result(
                subtask_type, subtask_parameters, self._handle_darwin_tarball)

    def _publish_darwin_debian(self):
        subtask_type = _DARWIN_DEBIAN_TASK_TYPE
        subtask_parameters = {
            build.YaPackage.AdhocPackagesParameter.name: [self._debian_pkg_json()],
        }
        subtask_parameters.update(self._debian_parameters('darwin'))
        SubtaskRunner(self).run_and_handle_result(subtask_type, subtask_parameters)

    def _common_parameters(self):
        return {
            build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
            build.parameters.UseArcadiaApiFuse.name: True,
            build.parameters.UseArcInsteadOfArcadiaApi.name: True,
        }

    def _build_parameters(self):
        parameters = self._common_parameters()
        parameters.update(yt_store_parameters())
        parameters.update({
            build.YaPackage.PackagesParameter.name: self.Parameters.building_pkg_json_path,
            build.YaPackage.CustomVersionParameter.name: self.Parameters.release_version,
            build.parameters.BuildSystem.name: 'semi_distbuild',
        })
        return parameters

    def _debian_parameters(self, platform):
        parameters = self._common_parameters()
        parameters.update({
            build.YaPackage.PackageTypeParameter.name: 'debian',
            build.YaPackage.SaveBuildOutputParameter.name: not self.Parameters.is_upload,
            build.YaPackage.PublishPackageParameter.name: self.Parameters.is_upload,
        })
        if self.Parameters.is_upload:
            parameters.update({
                build.YaPackage.PublishToParameter.name: 'mapsmobi-{}'.format(platform),
                build.YaPackage.ArchitectureAllParameter.name: True,
            })
        return parameters

    def _debian_pkg_json(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            with open(os.path.join(arcadia, self.Parameters.building_pkg_json_path)) as f:
                building_pkg_json = json.load(f)
        debian_pkg_json_object = {
            "meta": dict(building_pkg_json["meta"], version=self.Parameters.release_version),
            "data": [
                {
                    "source": {
                        "type": "SANDBOX_RESOURCE",
                        "id": self.Context.darwin_tarball_id,
                        "untar": True,
                    },
                    "destination": {
                        "path": "/",
                    }
                },
            ]
        }
        return json.dumps(debian_pkg_json_object, indent=4)

        resource.ttl = inf

    def _handle_linux_tarball(self, subtask):
        resource = get_ya_package_resource(subtask)
        resource.host_platform = 'linux'
        apply_navi_export_parameters(resource)

    def _handle_darwin_tarball(self, subtask):
        resource = get_ya_package_resource(subtask)
        resource.host_platform = 'darwin'
        apply_navi_export_parameters(resource)
        self.Context.darwin_tarball_id = resource.id
