import os
import shutil
import tarfile
import urllib
import logging

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

FILENAME_TEMPLATE = '{}_{}_amd64.deb'
URL_PREFIX = 'https://mirror.yandex.ru/ubuntu/pool/main/m/mesa/'
OPENGL_PACKAGES = ['mesa-common-dev', 'libgl1-mesa-glx', 'libgl1-mesa-dev']
ADDITIONAL_PACKAGES = [
    'http://mirror.yandex.ru/ubuntu/pool/main/libx/libxext/libxext6_1.3.3-1_amd64.deb',
    'http://mirror.yandex.ru/ubuntu/pool/main/libx/libx11/libx11-6_1.6.3-1ubuntu2.1_amd64.deb',
    'http://mirror.yandex.ru/ubuntu/pool/main/libx/libxcb/libxcb1_1.11.1-1ubuntu1_amd64.deb',
    'http://mirror.yandex.ru/ubuntu/pool/main/libx/libxau/libxau6_1.0.8-1_amd64.deb',
    'http://mirror.yandex.ru/ubuntu/pool/main/libx/libxdmcp/libxdmcp6_1.1.2-1.1_amd64.deb'
    ]


class MapsMobileOpengl(sdk2.Resource):
    target_os = sdk2.parameters.String('Target OS')
    version = sdk2.parameters.String('Version')


class MapsMobilePackageOpengl(sdk2.Task):
    ''' Task for packaging OpenGL for maps desktop libraries building. '''

    class Requirements(sdk2.Task.Requirements):
        privileged = True
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String('OpenGL packages version', required=True)

    def _init_paths(self):
        self._build_dir = 'build'
        self._download_dir = os.path.join(self._build_dir, 'download')
        self._install_dir = os.path.join(self._build_dir, 'install')
        for path in [self._download_dir, self._install_dir]:
            os.makedirs(path)

    def _get_package_urls(self, package):
        process = sp.Popen([
            'sudo',
            'apt-get',
            'install',
            '-qq',
            '--print-uris',
            '{}={}'.format(package, self.Parameters.version)
            ], stdout=sp.PIPE)
        out, _ = process.communicate()
        urls = [line.split(' ')[0][1:-1] for line in out.splitlines()]
        return urls;

    def _build_url_list(self):
        url_set = set()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("apt_update")) as pl:
            sp.check_call(
                ['sudo', 'apt-get', 'update'],
                stdout=pl.stdout,
                stderr=pl.stdout
                )

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("apt_clean")) as pl:
            sp.check_call(
                ['sudo', 'apt-get', 'clean'],
                stdout=pl.stdout,
                stderr=pl.stdout
                )

        for package in OPENGL_PACKAGES:
            package_urls = set(self._get_package_urls(package))
            url_set |= package_urls

        url_set |= set(ADDITIONAL_PACKAGES)

        return list(url_set)

    def _extract_package(self, filename):
        sp.Popen(['dpkg','-x', filename, self._install_dir]).wait()

    def _download_package(self, url):
        logging.info('Downloading \'{}\''.format(url))
        temp_name = urllib.urlretrieve(url)[0]
        shutil.move(temp_name, self._download_dir)
        return os.path.join(self._download_dir, os.path.basename(temp_name))

    def _make_opengl_resource(self):
        resource = MapsMobileOpengl(self, 'OpenGL version {}'.format(self._version), 'opengl.tar.gz', ttl='inf')
        resource.version = self._version
        resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(resource_data.path), 'w:gz') as tar:
            for entry in os.listdir(self._install_dir):
                tar.add(os.path.join(self._install_dir, entry), arcname=entry)
        resource_data.ready()

    def on_execute(self):
        self._version = self.Parameters.version
        self._init_paths()
        url_list = self._build_url_list()

        for url in url_list:
            filename = self._download_package(url)
            self._extract_package(filename)

        self._make_opengl_resource()
