import os
import shutil
import re
import tarfile
import logging

from sandbox import sdk2
from sandbox import common
from sandbox.common.errors import TaskError
import sandbox.common.types.client as ctc
from sandbox.sdk2.helpers import subprocess as sp


QT_PACKAGE_NAME_TEMPLATE='qt-opensource-{system_name}-{version}'
COMPILER_SUFFIXES = {'linux': 'gcc_64', 'darwin': 'clang_64'}
MODULE_PREFIXES = {'linux' : 'Qt5', 'darwin': 'Qt'}
PACKAGE_TYPES = {'linux' : 'run', 'darwin': 'dmg'}
SYSTEM_NAMES = {'linux': 'linux-x64', 'darwin': 'mac-x64-clang'}


class MapsMobileQtInstaller(sdk2.Resource):
    target_os = sdk2.parameters.String('Target OS')
    version = sdk2.parameters.String('Version')


class MapsMobileQt(sdk2.Resource):
    target_os = sdk2.parameters.String('Target OS')
    version = sdk2.parameters.String('Version')


def _output_precompiled(file_name):
    path = os.path.join(os.path.dirname(__file__), file_name)
    if not common.system.inside_the_binary():
        return path

    from library.python import resource
    resource_data = resource.find(path)
    if resource_data is None:
        raise TaskError('Precompiled resource not found: ' + file_name)

    with open(file_name, 'w') as f:
        f.write(resource_data)
    return os.path.abspath(file_name)


def _apply_patch(work_dir, patch):
    sp.Popen(['patch', '-p0', '--forward', '-i', patch], cwd=work_dir).wait()


def _type_based_copy(src, dst_dir):
    dst = os.path.join(dst_dir, os.path.basename(src))
    if os.path.islink(src):
        os.symlink(os.path.relpath(os.path.realpath(src), os.path.dirname(src)), dst)
    elif os.path.isdir(src):
        shutil.copytree(src, dst, symlinks=True)
    else:
        shutil.copy(src, dst)


def _copy_targets(src_dir, dst_dir, targets):
    for entry in os.listdir(src_dir):
        entry_path = os.path.join(src_dir, entry)
        for target in targets:
            if re.search(target, entry):
                _type_based_copy(entry_path, dst_dir)


def _copy_qt_subset(src_dir, dst_dir, libs, module_prefix, qt_modules, tools):
    os.makedirs(dst_dir)
    source_lib_dir = os.path.join(src_dir, 'lib')
    source_include_dir = os.path.join(src_dir, 'include')
    source_tools_dir = os.path.join(src_dir, 'bin')
    lib_dir = os.path.join(dst_dir, 'lib')
    include_dir = os.path.join(dst_dir, 'include')
    tools_dir = os.path.join(dst_dir, 'bin')
    os.mkdir(lib_dir)
    os.mkdir(include_dir)
    os.mkdir(tools_dir)

    _copy_targets(source_lib_dir, lib_dir, libs)
    _copy_targets(source_lib_dir, lib_dir, [module_prefix + module + '\\.' for module in qt_modules])
    _copy_targets(source_include_dir, include_dir, libs)
    _copy_targets(source_include_dir, include_dir, ['Qt' + module + '$' for module in (qt_modules + ['PlatformHeaders'])])
    _copy_targets(source_tools_dir, tools_dir, tools)
    _type_based_copy(os.path.join(src_dir, 'plugins'), dst_dir)
    _type_based_copy(os.path.join(src_dir, 'mkspecs'), dst_dir)
    _type_based_copy(os.path.join(src_dir, 'qml'), dst_dir)
    _type_based_copy(os.path.join(src_dir, 'lib', 'cmake'), os.path.join(dst_dir, 'lib'))


class MapsMobilePackageQt(sdk2.Task):
    ''' Task for packaging Qt for maps desktop libraries building. '''

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String('Qt version', required=True)
        with sdk2.parameters.String('Target OS', required=True) as target_os:
            target_os.values.linux = target_os.Value('linux', default=True)
            target_os.values.darwin = 'darwin'
        modules = sdk2.parameters.List('List of Qt modules')
        tools = sdk2.parameters.List('List of Qt tools')
        libs = sdk2.parameters.List('List of Qt dependencies')

    def _init_paths(self):
        target_os = self.Parameters.target_os
        system_name = SYSTEM_NAMES[target_os]
        compiler_suffix = COMPILER_SUFFIXES[target_os]
        self._extractor_path = _output_precompiled('extract-qt-installer')
        self._patch_path = _output_precompiled('darwin_xcode_8.patch')
        self._conf_path = _output_precompiled('qt.conf')
        self._version = self.Parameters.version
        self._package_type = PACKAGE_TYPES[target_os]
        self._module_prefix = MODULE_PREFIXES[target_os]
        self._package_name = QT_PACKAGE_NAME_TEMPLATE.format(
                system_name=system_name,
                version=self._version)
        self._short_version = '.'.join(self._version.split('.')[:2])
        self._build_dir = 'build'
        self._install_dir = os.path.join(self._build_dir, 'install')
        self._src_bin_dir = os.path.join(self._install_dir, self._short_version, compiler_suffix)
        self._dst_bin_dir = os.path.join(self._build_dir, 'qt')

    def _download_qt(self):
        resource = sdk2.Resource.find(
                MapsMobileQtInstaller,
                attrs={'target_os': self.Parameters.target_os, 'version': self._version}
                ).first()
        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        return str(resource_path)

    def _run_qt_extractor(self, extractor_script, installer_location, dst_dir):
        os.makedirs(dst_dir)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("qt_extractor")) as pl:
            sp.Popen([extractor_script, installer_location, os.path.abspath(dst_dir)], stdout=pl.stdout, stderr=sp.STDOUT).wait()

    def _install_qt_linux(self, downloaded_file):
        self._run_qt_extractor(self._extractor_path, downloaded_file, self._install_dir)

    def _mount_dmg(self, src, mount_point):
        os.makedirs(mount_point)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("mount")) as pl:
            sp.Popen(['hdiutil','attach', '-mountpoint', mount_point, src], stdout=pl.stdout, stderr=sp.STDOUT).wait()

    def _unmount_dmg(self, mount_point):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("unmount")) as pl:
            sp.Popen(['hdiutil', 'detach', '-force',  mount_point], stdout=pl.stdout, stderr=sp.STDOUT).wait()

    def _install_qt_darwin(self, downloaded_file):
        mount_point = os.path.join(self._build_dir, 'mount')
        installer_path = os.path.join(mount_point, self._package_name + '.app', 'Contents', 'MacOS', self._package_name)
        self._mount_dmg(downloaded_file, mount_point)
        try:
            self._run_qt_extractor(self._extractor_path, installer_path, self._install_dir)
        finally:
            self._unmount_dmg(mount_point)

    def _install_qt(self, downloaded_file):
        if self.Parameters.target_os == 'darwin':
            self._install_qt_darwin(downloaded_file)
        elif self.Parameters.target_os == 'linux':
            self._install_qt_linux(downloaded_file)
        else:
            raise TaskError('Unsupported system: ' + target_os)

    def _make_qt_resource(self):
        target_os = self.Parameters.target_os
        resource = MapsMobileQt(self, 'Qt version {} for {}'.format(self._version, target_os), 'qt.tar.gz', ttl='inf')
        resource.version = self._version
        resource.target_os = target_os
        resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(resource_data.path), 'w:gz') as tar:
            for entry in os.listdir(self._dst_bin_dir):
                tar.add(os.path.join(self._dst_bin_dir, entry), arcname=entry)
        resource_data.ready()

    def on_save(self):
        if self.Parameters.target_os == 'darwin':
            self.Requirements.client_tags = ctc.Tag.OSX_MOJAVE
            self.Requirements.container_resource = None
        else:
            self.Requirements.client_tags = ctc.Tag.LINUX_XENIAL
            self.Requirements.container_resource = 1165583468

    def on_execute(self):
        self._init_paths()
        downloaded_file = self._download_qt()
        self._install_qt(downloaded_file)

        if self.Parameters.target_os == 'darwin':
            _apply_patch(self._src_bin_dir, self._patch_path)

        libs = self.Parameters.libs
        qt_modules = self.Parameters.modules
        tools = self.Parameters.tools
        _copy_qt_subset(self._src_bin_dir, self._dst_bin_dir, libs, self._module_prefix, qt_modules, tools)
        _type_based_copy(self._conf_path, os.path.join(self._dst_bin_dir, 'bin'))

        self._make_qt_resource()
