import logging
import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common.arcadia import sdk as arc_sdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date, extract_branch_name

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.projects.maps.mobile.utils.testpalm_constants as tpc

_PALMSYNC_WRAPPER_BUILD_PATH = 'maps/mobile/tools/palmsync_wrapper'
_REQUIRED_THREADS = 16  # Will be enough to build a wrapper in a few minutes
_FIRST_BRANCH_WITH_TESTPALM_SUPPORT = '2021063000'  # Doesn't exist yet; TODO: selivni@: update it
_VALIDATION_ACTION = 'validate'
_SYNCHRONIZATION_ACTION = 'synchronize'


class MapsMobilePalmsyncAction(sdk2.Task):
    """
    Task for validating and synchronizing maps/mobile/libs
    Palmsync project against its remote TestPalm counterpart.
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_BIONIC

        # This fixes connecting to TestPalm servers
        dns = ctm.DnsType.DNS64

        cores = _REQUIRED_THREADS
        ram = 8192  # Might need to increase it if Palmsync doesn't handle a big project

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.String('Palmsync action', required=True) as action:
            action.values.validate = action.Value(_VALIDATION_ACTION, default=True)
            action.values.synchronize = _SYNCHRONIZATION_ACTION

        with sdk2.parameters.Group('Repository'):
            arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
            use_svn = sdk2.parameters.Bool('Use SVN', default=True, required=False)
            with use_svn.value[True]:
                arcadia_revision = sdk2.parameters.String('Arcadia revision', required=True)

        with action.value[_SYNCHRONIZATION_ACTION]:
            testpalm_token = sdk2.parameters.YavSecret(
                'TestPalm oauth token of an owner of MapKit TestPalm projects',
                required=True)
            test_run = sdk2.parameters.Bool(
                'Test run',
                description='Run synchronization on a mock_mapkit_test project',
                default=False,
                required=False)

        with action.value[_VALIDATION_ACTION]:
            arcadia_patch = sdk2.parameters.String('Apply patch', required=False)

    def on_execute(self):
        # We shouldn't attempt to launch this task for branches that don't have their own TestPalm project
        if not is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_TESTPALM_SUPPORT):
            raise TaskError('Attempting to launch Palmsync action on a branch without TestPalm project!')
        with arc_sdk.mount_arc_path(
                self.Parameters.arcadia_url
                    if not self.Parameters.use_svn
                    else '{}@{}'.format(self.Parameters.arcadia_url, self.Parameters.arcadia_revision),
                use_arc_instead_of_aapi=True) as mount_path:
            self._mount_path = mount_path
            self._add_result_dir = os.getcwd()

            # For some reason, unlike in local builds, there is no symlink created in
            # arcadia/maps/mobile/tools/palmsync_wrapper. I have to specify results_dir.
            # However, `--add-result` gets handled in a weird manner: the binary seems to appear in
            # result_dir/maps/mobile/tools/palmsync_wrapper. This is why the path below exists.
            self._palmsync_wrapper_binary_path = os.path.join(
                self._add_result_dir,
                'maps',
                'mobile',
                'tools',
                'palmsync_wrapper',
                'palmsync_wrapper')
            self._mapsmobi_project_path = os.path.join(
                self._mount_path,
                'maps',
                'mobile',
                'libs')
            if self.Parameters.arcadia_patch:
                arc_sdk.apply_patch(
                    task=self,
                    arcadia_path=self._mount_path,
                    arcadia_patch=self.Parameters.arcadia_patch,
                    dest_dir=os.getcwd())
            self._build_palmsync_wrapper()

            # Add Filename tag to all .yml files
            self._add_filenames_to_cases()

            # Palmsync doesn't validate before synchronizing for some reason,
            # so we have to do that manually.
            self._validate()
            if self.Parameters.action == _SYNCHRONIZATION_ACTION:
                self._synchronize()

    def _build_palmsync_wrapper(self):
        arc_sdk.do_build(
            build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
            source_root=self._mount_path,
            targets=[_PALMSYNC_WRAPPER_BUILD_PATH],
            results_dir=self._add_result_dir,
            add_result=['palmsync_wrapper'],
            clear_build=False,
            build_threads=_REQUIRED_THREADS,
            sandbox_token=self.server.task_token)

    # Search maps/mobile/libs for .yml files in testpalm dirs, write
    # relative path of them to the beginning of each file
    def _add_filenames_to_cases(self):
        for dir_tuple in os.walk(self._mapsmobi_project_path):
            directory = dir_tuple[0]
            if os.path.basename(directory) == 'testpalm':
                for filename_end in os.listdir(directory):
                    if not filename_end.endswith('.yml'):
                        continue
                    absolute_filename = os.path.join(directory, filename_end)
                    relative_filename = os.path.relpath(
                        absolute_filename,
                        self._mapsmobi_project_path)
                    with open(absolute_filename, 'r+') as fileobject:
                        content = fileobject.read()
                        fileobject.seek(0, 0)
                        fileobject.write(
                            'Filename: "{filename}"'.format(filename=relative_filename) + '\n' + content)

    def _setup_env(self):
        env_copy = os.environ
        env_copy['palmsync_testpalmToken'] = self.Parameters.testpalm_token.data()['testpalm_token']
        branch_name = extract_branch_name(self.Parameters.arcadia_url)
        if self.Parameters.action == _SYNCHRONIZATION_ACTION and not self.Parameters.test_run:
            if branch_name == 'trunk':
                env_copy['palmsync_project'] = tpc.TRUNK_TESTPALM_PROJECT
            else:
                env_copy['palmsync_project'] = tpc.RELEASE_BRANCH_TESTPALM_PROJECT_TEMPLATE.format(branch_name=branch_name)

    def _validate(self):
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('palmsync_validate')) as pl:
            sdk2.helpers.subprocess.check_call(
                [self._palmsync_wrapper_binary_path, _VALIDATION_ACTION],
                cwd=self._mapsmobi_project_path,
                stdout=pl.stdout,
                stderr=pl.stderr)

    def _synchronize(self):
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('palmsync_synchronize')) as pl:
            sdk2.helpers.subprocess.check_call(
                [self._palmsync_wrapper_binary_path, _SYNCHRONIZATION_ACTION],
                cwd=self._mapsmobi_project_path,
                env=self._setup_env(),
                stdout=pl.stdout,
                stderr=pl.stderr)
