import os

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import notification as ctn
import sandbox.common.types.task as task_types
from sandbox.projects.common.vcs import arc
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner


# make test_app and androidTest apks
class MapsMobilePerftest(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        notifications = (
            sdk2.Notification(
                (task_types.Status.FAILURE, task_types.Status.EXCEPTION),
                ('geoapps-infra-notify@yandex-team.ru', 'imseleznev'),
                ctn.Transport.EMAIL
            ),
        )

        commit = sdk2.parameters.String(
            'Commit',
            description='An example:4a9dbeabf69391c9ee3ecce89580e43ba315e525',
            default='')

        branch = sdk2.parameters.String(
            'Branch name',
            description='An example:trunk, users/<username>/branch',
            default='trunk', required=True)

        # Output parameters
        with sdk2.parameters.Output:
            result_commit = sdk2.parameters.String('Resulting commit', required=True)
            app_apk = sdk2.parameters.Resource("TestApp apk", required=True)
            instrumented_tests_apk = sdk2.parameters.Resource("InstrumentedTests apk", required=True)

    class Context(sdk2.Context):
        commit = None

    def _resolve_result_commit(self):
        arc_client = arc.Arc()
        with arc_client.mount_path(None, None, fetch_all=False) as arcadia_src_dir:
            arc_client.checkout(
                arcadia_src_dir,
                branch=self.Parameters.branch,
            )
            if not self.Parameters.commit:
                # get HEAD commit hash
                return arc_client.log(
                    arcadia_src_dir,
                    start_commit="HEAD",
                    max_count=1,
                    as_dict=True
                )[0]['commit']
            else:
                # check if commit exists in branch
                merge_base = arc_client.get_merge_base(
                    arcadia_src_dir,
                    base_branch=self.Parameters.commit,
                    branch=self.Parameters.branch,
                )
                if merge_base == self.Parameters.commit:
                    return self.Parameters.commit
                else:
                    raise common.errors.TaskError(
                        'Could not find commit {} in'
                        ' branch {}'.format(
                            self.Parameters.commit,
                            self.Parameters.branch)
                    )

    def _get_params(self):
        parameters = {
            'prebuilt_bundle': False,
            'package_config': 'maps/mobile/bundle/packages/everything-android.pkg.json',
            'build_variant': 'release',
            'app_src_path': 'maps/mobile/apps/test_app/android',
            'upload_to_beta': False,
            'jdk_resource': 810373574,
            'android_sdk_resource': 2431170592,
            'gradle_resource': 2566840956,
            'maven_repo_resource': 2586526945,
            'keystore': 'sec-01dvx8g7d7bbxzff6d61t44jek',
            'custom_apk_path': os.path.join('build', 'outputs', 'apk', 'ym', 'release')
        }
        return parameters

    def _build_test_app(self, parameters):
        with self.memoize_stage.run_test_app:
            SubtaskRunner(self).run('MAPS_MOBILE_BUILD_ANDROID_APP', parameters)
        test_app_subtask = SubtaskRunner(self).find_task('MAPS_MOBILE_BUILD_ANDROID_APP')
        SubtaskRunner.require_successful(test_app_subtask)
        with self.memoize_stage.get_resources:
            self.Parameters.app_apk = test_app_subtask.Parameters.artifact_resource
        self._bundle_resource = test_app_subtask.Parameters.output_bundle_resource

    def _build_instrumented_app(self, parameters):
        with self.memoize_stage.run_instrumented_app:
            SubtaskRunner(self).run('MAPS_MOBILE_BUILD_ANDROID_APP', parameters)
        instrumented_subtask = SubtaskRunner(self).find_task('MAPS_MOBILE_BUILD_ANDROID_APP')
        SubtaskRunner.require_successful(instrumented_subtask)
        with self.memoize_stage.set_instrumented_apk:
            self.Parameters.instrumented_tests_apk = instrumented_subtask.Parameters.artifact_resource

    def on_execute(self):
        parameters = self._get_params()

        with self.memoize_stage.resolve_commit:
            self.Context.commit = self._resolve_result_commit()
        parameters.update({'arcadia_url': 'arcadia-arc:/#{}'.format(
            self.Context.commit), 'use_svn': False})

        self._build_test_app(parameters)

        parameters.update({
            'prebuilt_bundle': True,
            'bundle_resource': self._bundle_resource,
            'custom_build_command': 'assembleAndroidTest',
            'custom_apk_path': os.path.join('build', 'outputs', 'apk', 'androidTest', 'ym', parameters['build_variant'])
        })
        self._build_instrumented_app(parameters)

        with self.memoize_stage.set_result_commit:
            self.Parameters.result_commit = self.Context.commit
