import os
import json
import zipfile
import logging
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskError
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource

_ARTIFACTS_PATH = 'artifacts'

_AAR_NAME = os.path.join(_ARTIFACTS_PATH, 'bundle/maps.mobile.aar')

_ARCHS = {
    'arm32': 'armeabi-v7a',
    'arm64': 'arm64-v8a',
    'x86': 'x86',
    'x86_64': 'x86_64'
}

# Remove 'libcom.yandex.maps.mobile.so' after several releases after this commit.
_POSSIBLE_BASENAMES = [
    'libcom.yandex.maps.mobile.so',
    'libmaps-mobile.so',
]

_BUNDLES = [
    'auth',
    'recording',
    'datasync',
    'mapkit',
    'directions',
    'places',
    'search',
    'transport',
    'bookmarks',
    'push',
    'mrc',
]

YT_BASE_DIR = '//home/maps-mobile-ci/aar-size'
YT_TABLE_NAME = 'statistics_testenv'
YT_TABLE_PATH = YT_BASE_DIR + '/' + YT_TABLE_NAME

YT_TOKEN_ID = 'sec-01e1acjhzr9d6kdwzqaxrrw78t'


def path_to_so(arch):
    for basename in _POSSIBLE_BASENAMES:
        filename = os.path.join('jni', arch, basename)
        if os.path.isfile(filename):
            return filename
    raise TaskError('Task has not found a .so file to evaluate for arch {}'.format(arch))


def yt_create_table(yt):
    if not yt.exists(YT_TABLE_PATH):
        schema = [
            {"name": "revision", "type": "uint64"},
            {"name": "type", "type": "string"},
            {"name": "size", "type": "uint64"},
            {"name": "date", "type": "string"}
        ]
        yt.create("table", YT_TABLE_PATH, attributes={'schema': schema, 'dynamic': False})

def combine_chunks(yt):
    info = json.loads(yt.list(YT_BASE_DIR, format='json', attributes=['chunk_count']))
    if len(info) and info[0]['$value'] == YT_TABLE_NAME:
        chunk_count = info[0]['$attributes']['chunk_count']
        print("Chunk count of table is ", chunk_count)
        if chunk_count > 20:
            yt.run_merge(YT_TABLE_PATH, YT_TABLE_PATH, mode='auto', spec={'combine_chunks': True})
            print("Combine chunks is completed")


class MapsMobilePublishSoSize(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet")
        ]

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Arcadia URL:',
            description='Example with a specified revision: arcadia:/arc/trunk/arcadia@6150000',
            required=True)

    def _get_commit_date(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            return arcadia_sdk.mounted_path_svnversion(arcadia)['date'].split('T')[0]

    def _make_artifacts(self):
        with self.memoize_stage.run_subtask:
            parameters = {
                'arcadia_url': self.Parameters.arcadia_url,
                'libraries': _BUNDLES,
                'platform': 'android',
                'is_navi': False
            }
            SubtaskRunner(self).run('MAPS_MOBILE_MAKE_ARTIFACTS', parameters)
        subtask = SubtaskRunner(self).find_task('MAPS_MOBILE_MAKE_ARTIFACTS')
        SubtaskRunner.require_successful(subtask)
        self.artifacts = subtask.Parameters.artifacts

    def on_execute(self):
        self._make_artifacts()
        extract_resource(self.artifacts, _ARTIFACTS_PATH)

        with zipfile.ZipFile(_AAR_NAME, 'r') as zip_ref:
            zip_ref.extractall()

        import yt.wrapper as yt
        yt_token = sdk2.yav.Secret(YT_TOKEN_ID).data()['yt-token']
        yt_client = yt.YtClient('hahn', token=yt_token)

        yt_create_table(yt_client)

        rows = []
        commit_date = self._get_commit_date()
        revision = int(self.Parameters.arcadia_url.split('@')[1])
        for key in sorted(_ARCHS):
            filename = path_to_so(_ARCHS[key])
            stat = os.stat(filename)
            row = {'type': key,
                   'size': stat.st_size,
                   'revision': revision,
                   'date': commit_date}
            rows.append(row)
        yt_client.write_table(yt.TablePath(YT_TABLE_PATH, append=True), rows)

        combine_chunks(yt_client)
