import hashlib
import logging
import os
import shutil
import tempfile

import boto3

from sandbox import sdk2
from sandbox.common.errors import TaskError
import sandbox.common.types.client as ctc
from sandbox.sdk2.helpers import ProcessLog

from sandbox.projects.maps.common.retry import retry

from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileFrameworkZip


_SNAPSHOT_REPO = 'ssh://git.yandex.ru/maps/mapsmobi_pods.git'
_NON_SNAPSHOT_REPO = 'ssh://git@bitbucket.browser.yandex-team.ru/mcp/mobile-cocoa-pod-specs.git'
_MDS_HOST = 'https://s3.mds.yandex.net'

_PODSPEC_TEMPLATE = u'''\
Pod::Spec.new do |s|

  s.name = '{framework_name}'
  s.version = '{version}'
  s.summary = '{framework_name} iOS Framework'
  s.homepage = 'https://tech.yandex.com/maps/mapkit'
  s.license = {{ :type => 'Proprietary', :text => 'https://yandex.ru/legal/maps_api/ 2016 \u00A9 Yandex. All rights reserved.' }}
  s.authors = {{ 'Yandex LLC' => 'maps-api@support.yandex.ru' }}
  s.platform = :ios, '10.0'

  s.frameworks = {framework_dependencies}
  s.libraries = {library_dependencies}

  s.source = {{ :http => '{url}',
               :sha1 => '{sha1}' }}

  s.documentation_url = 'https://tech.yandex.com/maps/mapkit'
  s.resource = '{bundle_dir}'
  s.vendored_frameworks = '{framework_dir}'
  s.preserve_paths = '{framework_dir}'
end
'''


class MapsMobileReleaseCocoaPod(sdk2.Task):
    """ Task for releasing CocoaPod. """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_MAPS_MOBILE_DARWIN & ctc.Tag.USER_MAPS_MOBILE_DARWIN

    class Parameters(sdk2.Task.Parameters):

        framework_zip = sdk2.parameters.Resource(
                'Framework zip',
                required=True,
                resource_type=MapsMobileFrameworkZip,
                )
        framework_name = sdk2.parameters.String(
                'Framework name',
                required=True,
                )
        release_version = sdk2.parameters.String('Release version', required=True)
        framework_dependencies = sdk2.parameters.List('Platform framework dependencies for framework')
        library_dependencies = sdk2.parameters.List('Platform library dependencies for framework')
        s3_credentials = sdk2.parameters.YavSecret(
                's3 mds key secret',
                default='sec-01e20smy30qbgz6cs9abgtc4ma',
                required=True
                )
        git_user_name = sdk2.parameters.String('Git user name', default='robot-mapkit-ci', required=True)
        git_email = sdk2.parameters.String('Git e-mail', default='robot-mapkit-ci@yandex-team.ru', required=True)
        git_ssh_vault_owner = sdk2.parameters.String(
                'Git ssh Key Sandbox Vault owner',
                default='robot-mapkit-ci',
                required=True
                )
        git_ssh_vault_name = sdk2.parameters.String(
                'Git ssh Key Sandbox Vault name',
                default='ssh_key',
                required=True
                )
        framework_dir = sdk2.parameters.String(
                'Path to framework/xcframework inside archive',
                required=True
                )
        bundle_dir = sdk2.parameters.String(
                'Path to resource bundle inside archive',
                required=True)
        # Public release functions are disabled until thorough testing. MAPSMOBCORE-10692
        # is_public = sdk2.parameters.Bool('Is public')
        is_snapshot = sdk2.parameters.Bool('Is snapshot', default=False)

    def on_execute(self):
        self._framework_zip_path = str(sdk2.ResourceData(self.Parameters.framework_zip).path)
        self._calculate_sha1()
        self._upload_to_mds()
        self._generate_podspec()
        self._commit_podspec()

    def _calculate_sha1(self):
        with open(self._framework_zip_path, 'rb') as zip_:
            zip_bytes = zip_.read()
            self._sha1 = hashlib.sha1(zip_bytes).hexdigest()

    def _upload_to_mds(self):
        file_key = '{framework_name}-{podspec_version}.framework.zip'.format(
                framework_name=self.Parameters.framework_name,
                podspec_version=self.Parameters.release_version,
                )
        # Public release functions are disabled until thorough testing. MAPSMOBCORE-10692
        bucket_name = ('maps-ios-pods-public'
                       if False  # self.Parameters.is_public
                       else 'maps-ios-pods-internal')
        mds_credentials = self.Parameters.s3_credentials.data()
        s3_client = boto3.session.Session().client(
            service_name='s3',
            endpoint_url=_MDS_HOST,
            aws_access_key_id=mds_credentials['key_id'],
            aws_secret_access_key=mds_credentials['key'],
            )
        s3_client.upload_file(
                Filename=self._framework_zip_path,
                Bucket=bucket_name,
                Key=file_key,
                )
        # Public release functions are disabled until thorough testing. MAPSMOBCORE-10692
        mds_url_format = ('https://{bucket_name}.s3.yandex.net/{file_key}'
                          if False  # self.Parameters.is_public
                          else 'https://s3.mds.yandex.net/{bucket_name}/{file_key}')
        self.Context.mds_url = mds_url_format.format(bucket_name=bucket_name,
                                                     file_key=file_key)

    def _generate_podspec(self):
        self._podspec = _PODSPEC_TEMPLATE.format(
                bundle_dir=self.Parameters.bundle_dir,
                framework_dir=self.Parameters.framework_dir,
                framework_name=self.Parameters.framework_name,
                version=self.Parameters.release_version,
                url=self.Context.mds_url,
                sha1=self._sha1,
                framework_dependencies=','.join({"'{}'".format(framework) for framework in self.Parameters.framework_dependencies}),
                library_dependencies=','.join({"'{}'".format(library) for library in self.Parameters.library_dependencies}),
            )

    @retry(tries=5, delay=8, backoff=2)
    def _commit_podspec(self):
        with sdk2.ssh.Key(self, self.Parameters.git_ssh_vault_owner, self.Parameters.git_ssh_vault_name):
            repo = _SNAPSHOT_REPO if self.Parameters.is_snapshot else _NON_SNAPSHOT_REPO
            git = sdk2.vcs.git.Git(repo)

            try:
                temp_dir = tempfile.mkdtemp()
                checkout_path = os.path.abspath(os.path.join(temp_dir, 'podspec_repo'))
                git.clone(checkout_path, 'master')
                podspec_dir = os.path.join(
                        checkout_path, self.Parameters.framework_name, self.Parameters.release_version
                        )
                podspec_fullname = os.path.join(
                        podspec_dir,
                        '{framework_name}.podspec'.format(framework_name=self.Parameters.framework_name)
                        )
                if os.path.exists(podspec_dir):
                    raise TaskError('This version {} already uploaded to {}!'
                                      .format(self.Parameters.release_version, repo))
                os.makedirs(podspec_dir)
                with open(podspec_fullname, 'wb') as f:
                    f.write(self._podspec)

                git.execute('config', '--local', 'user.name', self.Parameters.git_user_name, cwd=checkout_path)
                git.execute('config', '--local', 'user.email', self.Parameters.git_email, cwd=checkout_path)
                git.execute('add', podspec_fullname, cwd=checkout_path)
                commit_message = (
                        'Add podspec for {framework_name} version {release_version}'.format(
                        framework_name=self.Parameters.framework_name,
                        release_version=self.Parameters.release_version)
                )
                git.execute('commit', '-m', commit_message, cwd=checkout_path)
                git.execute('push', 'origin', 'master', cwd=checkout_path)
            finally:
                shutil.rmtree(temp_dir)
