import os
import tarfile
import logging
import shutil
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource

internalConfig = {
    'repo': 'http://artifactory.yandex.net/artifactory/yandex_mobile_releases',
    'snapshot_repo': 'http://artifactory.yandex.net/artifactory/yandex_mobile_snapshots'}

publicConfig = {'repo': 'https://oss.sonatype.org/service/local/staging/deploy/maven2/'}

ARTIFACTORY_TEMPLATE = '''project.configure(project) {{

    configurations {{
        bundle
    }}

    logger.lifecycle("uploading release to {repo} with version {version}")

    def baseName = '{base_name}'

    artifacts {{
        bundle(file('{base_name}.aar'))
    }}

    def sourcesFile = file('{base_name}-sources.jar')
    if (sourcesFile.exists()) {{
        artifacts {{
            bundle(sourcesFile) {{
                classifier = 'sources'
            }}
        }}
    }}

    def symbolsFile = file('{base_name}.symbols.aar')
    if (symbolsFile.exists()) {{
        artifacts {{
            bundle(symbolsFile) {{
                classifier = 'symbols'
            }}
        }}
    }}

    uploadBundle {{
        repositories.mavenDeployer {{

            repository(url: "{repo}") {{
                authentication(
                        userName: "{key}",
                        password: "{secret}")
            }}

            pom.project {{ //TODO: Delete this closure after several release cycles
                groupId "{group_id}"
                artifactId "{artifact_id}"
                version "{version}"
            }}

            pom.withXml {{
                def pomXmlNode = asNode()
                def pomFile = file('{base_name}-pom.xml')
                if (pomFile.exists()) {{
                    def pomFileNode = new XmlParser().parse(pomFile)
                    pomXmlNode.setValue(pomFileNode.value())
                    pomXmlNode.version[0].value = "{version}"
                    if ({isSnapshot} && pomXmlNode.dependencies.size() != 0)
                        pomXmlNode.remove(pomXmlNode.dependencies)
                }} else if (pomXmlNode.dependencies.size() != 0) {{ //TODO: Delete this branch after several release cycles
                    pomXmlNode.dependencies.get(0).get('dependency').each {{ dep ->
                        String artifactId = dep.get('artifactId').text()
                        if (dep.get('groupId').size() == 0) {{
                            String groupId = artifactId.tokenize('.')[0..-2].join('.')
                            dep.appendNode('groupId', groupId)
                            dep.get('artifactId')[0].value = artifactId.tokenize('.').last()
                        }}
                    }}
                }}
            }}
        }}
    }}
}}'''

def _move_from_bundle_to_curdir(base_name):
    for suffix in ['.symbols.aar', '.aar', '-pom.xml', '-sources.jar']:
        filename = base_name + suffix
        if os.path.isfile('bundle/' + filename):
            shutil.move('bundle/' + filename, filename)

# Task for upload maps.mobile.aar to artifactory.yandex.net
class MapsMobileReleaseMaven(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        aar_resource = sdk2.parameters.Resource('Artifacts resource', required=True)
        aar_base_name = sdk2.parameters.String('Artifact base name in archive', required=True)
        gradle_resource = sdk2.parameters.Resource(
                'Gradle resource',
                default=1069988336,
                required=True,
                )
        jdk_resource = sdk2.parameters.Resource(
                'JDK resource',
                default=810373574,
                required=True,
                )
        maven_resource = sdk2.parameters.Resource(
                'Maven repo resource',
                default=1274061471,
                required=True,
                )
        group_id = sdk2.parameters.String('Release group', required=True)
        artifact_id = sdk2.parameters.String('Release artifact', required=True)
        version = sdk2.parameters.String('Release version', required=True)

        # MAPSMOBCORE-10692
        is_public = False  # sdk2.parameters.Bool('Is public (False for internal)', required=True)
        # with is_public.value[False]:
        is_snapshot = sdk2.parameters.Bool('Is snapshot', required=True)

        artifacts_config = sdk2.parameters.YavSecret(
                    'Yav secret with artifacts key and secret',
                    default='sec-01e0r4j6fr2ks9jpnfy2a086px',
                    required=True,
                    )

    def _create_build_gradle(self):
        with open('build.gradle', 'w') as file:
            file.write('''apply plugin: 'maven'
                    apply from: "artifactory.gradle"
                    buildscript {
                        repositories {
                            maven {
                                url "%s"
                            }
                        }
                        dependencies {
                            classpath 'com.android.tools.build:gradle:1.3.1'
                        }
                    }
                    ''' % os.path.abspath(self._maven_dir))

    def _create_artifactory_gradle(self):
        config = publicConfig if self.Parameters.is_public else internalConfig
        repo = config['repo']
        version = self.Parameters.version
        isSnapshot = 'false'
        if not self.Parameters.is_public and self.Parameters.is_snapshot:
            isSnapshot = 'true'
            version += '-SNAPSHOT'
            repo = config['snapshot_repo']

        content = ARTIFACTORY_TEMPLATE.format(version = version,
            base_name = self.Parameters.aar_base_name,
            key = self.Parameters.artifacts_config.data()['key'],
            secret = self.Parameters.artifacts_config.data()['secret'],
            group_id = self.Parameters.group_id,
            artifact_id = self.Parameters.artifact_id,
            isSnapshot = isSnapshot,
            repo = repo)

        with open('artifactory.gradle', 'w') as file:
            file.write(content)

    def _setup_resources(self):
        extract_resource(self.Parameters.aar_resource, path='.')
        _move_from_bundle_to_curdir(self.Parameters.aar_base_name)

        extract_resource(self.Parameters.gradle_resource, path='.')

        self._maven_dir = os.path.join('build', 'maven')
        extract_resource(self.Parameters.maven_resource, self._maven_dir)

        self._jdk_dir = os.path.join('build', 'jdk')
        extract_resource(self.Parameters.jdk_resource, self._jdk_dir, gzip=False)

    def _release_maven(self):
        self._create_build_gradle()
        self._create_artifactory_gradle()
        gradle_env = os.environ
        gradle_env.update({'JAVA_HOME': os.path.abspath(self._jdk_dir)})
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("gradle_uploadBundle")) as pl:
            sp.Popen(['./bin/gradle', '--offline', 'uploadBundle'],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
                env=gradle_env).wait()

    def on_execute(self):
        self._setup_resources()
        self._release_maven()
