import json
import os
from glob import glob
import shutil

from sandbox.common.errors import TaskError, TaskFailure
from sandbox import sdk2
from sandbox.projects.common.task_env import TinyRequirements

from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.subprocess_helpers import process_subtask_status

TSR_TASK_TYPE = 'TEAMCITY_SANDBOX_RUNNER_LAUNCHER'
BUILD_LOGS_RESOURCE_TYPE = 'BUILD_LOGS'
BUILD_LOGS_RESOURCE_DESCRIPTION = 'TSR build logs'

REPORT_URLS_MESSAGE = """\
Build log: <a href="{proxy_url}/index.html">index.html</a>
Full build log: <a href="{proxy_url}/full_index.html">full_index.html</a>
"""


class MapsMobileTsrWrapper(sdk2.Task):
    class Requirements(TinyRequirements):
        disk_space = 10000

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Artifact parameters') as artifact_params:
            artifact_path = sdk2.parameters.String(
                'Path to artifact or directory inside TEAMCITY_ARTIFACTS resource (glob)',
                description='If empty artifact will not be created',
                default='')
            artifact_resource_type = sdk2.parameters.String('Artifact resource type', default='OTHER_RESOURCE')
            artifact_resource_description = sdk2.parameters.String(
                'Artifact resource description',
                default='TSR artifact')
            artifact_attrs = sdk2.parameters.Dict('Artifact attrs', default={})
            build_logs_ttl = sdk2.parameters.Integer('Build logs ttl', default=1)

        with sdk2.parameters.Group('Teamcity sandbox runner vcs parameters') as tsr_common_params:
            commit = sdk2.parameters.String(
                'Commit', )
            ssh_key = sdk2.parameters.Vault(
                'SSH key', )

        with sdk2.parameters.Group('Service parameters') as service_params:
            force_clean_build = sdk2.parameters.Bool(
                'Force clean build',
                description='if True, cache archives will be reassembled',
                default=False
            )

        with sdk2.parameters.Group('Teamcity sandbox runner config parameters') as tsr_config_params:
            config_from_repository = sdk2.parameters.Bool(
                'take config from repository',
                default=True, )
            with config_from_repository.value[False]:
                config = sdk2.parameters.String(
                    'Config (yaml)',
                    multiline=True, )
            with config_from_repository.value[True]:
                config_path = sdk2.parameters.String(
                    'Config path', )
            env = sdk2.parameters.Dict('env', default={})
            secrets = sdk2.parameters.Dict('secrets', default={})

    def on_execute(self):
        subtask_runner = SubtaskRunner(self)

        with self.memoize_stage['run_tsr']:
            subtask_runner.run(
                TSR_TASK_TYPE,
                parameters={
                    'commit': self.Parameters.commit,
                    'ssh_key': self.Parameters.ssh_key,
                    'force_clean_build': self.Parameters.force_clean_build,
                    'config_from_repository': self.Parameters.config_from_repository,
                    'config': self.Parameters.config,
                    'config_path': self.Parameters.config_path,
                    'repo_url': 'arcadia',
                    'env': json.dumps(self.Parameters.env),
                    'secrets': json.dumps(self.Parameters.secrets)
                },
                wait=True
            )

        subtasks = subtask_runner.find_tasks()
        assert len(subtasks) == 1
        subtask = subtasks[0]

        tsr_artifacts = sdk2.Resource['TEAMCITY_ARTIFACTS'].find(task=subtask).first()
        if not tsr_artifacts:
            raise TaskFailure("TEAMCITY_ARTIFACTS resource not found in child task")

        tsr_artifacts_path = sdk2.ResourceData(tsr_artifacts).path

        logs_resource = sdk2.Resource[BUILD_LOGS_RESOURCE_TYPE](
            task=self,
            description=BUILD_LOGS_RESOURCE_DESCRIPTION,
            path='build_logs',
            ttl=self.Parameters.build_logs_ttl)

        logs_data = sdk2.ResourceData(logs_resource)
        os.makedirs(str(logs_data.path))
        for log_file in ('index.html', 'full_index.html'):
            shutil.copy(str(tsr_artifacts_path / log_file), str(logs_data.path))
        logs_data.ready()

        self.set_info(
            REPORT_URLS_MESSAGE.format(proxy_url=logs_resource.http_proxy),
            do_escape=False)

        process_subtask_status(subtask.status, "Subtask failed. See index.html or full_index.html for details")

        if self.Parameters.artifact_path:
            artifacts = list(tsr_artifacts_path.glob(self.Parameters.artifact_path))
            if len(artifacts) == 0:
                raise TaskError('Artifact path not found in TEAMCITY_ARTIFACTS resource')
            if len(artifacts) > 1:
                raise TaskError('More then one artifact found in TEAMCITY_ARTIFACTS resource')

            artifact = str(artifacts[0])
            if os.path.isdir(artifact):
                path = 'artifacts'
            else:
                path = os.path.basename(artifact)

            target_resource = sdk2.Resource[self.Parameters.artifact_resource_type](
                task=self,
                description=self.Parameters.artifact_resource_description,
                path=path,
                **self.Parameters.artifact_attrs)

            resource_data = sdk2.ResourceData(target_resource)
            if os.path.isdir(artifact):
                shutil.copytree(artifact, str(resource_data.path), symlinks=True)
            else:
                shutil.copy(artifact, str(resource_data.path))
            resource_data.ready()
