import logging
import os
import shutil
import zipfile

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk as arc_sdk
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource
import sandbox.common.types.misc as ctm

_REQUIRED_THREADS = 16

_GRADLE_PROPERTY_PREFIX = 'ORG_GRADLE_PROJECT_'
_ANDROID_TESTAPP_PATH = os.path.join('maps', 'mobile', 'apps', 'test_app', 'android')

_FIRST_BRANCH_WITH_FULL_SYMBOLS = '2021042919'


class MapsMobileUploadAndroidCrashlytics(sdk2.Task):
    """
    Task for generating and uploading symbol files to Crashlytics.
    Requires an archive containing two AARs: maps.mobile.aar and
    maps.mobile.symbols.aar
    """
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        cores = _REQUIRED_THREADS
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        artifacts = sdk2.parameters.Resource(
            'MapKit artifacts',
            resource_type=resource_types.YA_PACKAGE,
            required=True,
            description='An archive with maps.mobile.(symbols.)aar inside')
        jdk_resource = sdk2.parameters.Resource('JDK resource', required=True)
        android_sdk_resource = sdk2.parameters.Resource('Android SDK resource', required=True)
        with sdk2.parameters.Group('Repository'):
            arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
            use_svn = sdk2.parameters.Bool('Use SVN', default=True, required=False)
            with use_svn.value[True]:
                arcadia_revision = sdk2.parameters.String('Arcadia revision', required=True)
        with sdk2.parameters.Output:
            same_artifacts = sdk2.parameters.Resource(
                'MapKit artifacts for testenv app build',
                resource_type=resource_types.YA_PACKAGE)

    def on_execute(self):
        self._unpack_dir = os.getcwd()
        self._unstripped_aar_dir = os.path.abspath(os.path.join(self._unpack_dir, 'unstripped_aar'))
        self._stripped_aar_dir = os.path.abspath(os.path.join(self._unpack_dir, 'stripped_aar'))

        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_FULL_SYMBOLS):
            self._extract_resources()
            with arc_sdk.mount_arc_path(
                    self.Parameters.arcadia_url
                        if not self.Parameters.use_svn
                        else '{}@{}'.format(self.Parameters.arcadia_url, self.Parameters.arcadia_revision),
                    use_arc_instead_of_aapi=True) as mount_path:
                self._android_testapp_path = os.path.join(
                    mount_path,
                    _ANDROID_TESTAPP_PATH)
                self._set_gradle_env()
                self._run_upload()
        artifacts_path = str(sdk2.ResourceData(self.Parameters.artifacts).path)
        artifacts_copy_path = os.path.join(
            os.getcwd(),
            os.path.basename(artifacts_path))
        shutil.copy(artifacts_path, artifacts_copy_path)
        self.Parameters.same_artifacts = resource_types.YA_PACKAGE(
            self,
            'MapKit artifacts',
            artifacts_copy_path,
            ttl=31)
        sdk2.ResourceData(self.Parameters.same_artifacts).ready()

    def _extract_resources(self):
        extract_resource(self.Parameters.artifacts, self._unpack_dir)

        unstripped_aar_path = os.path.abspath(
                os.path.join(self._unpack_dir, 'maps.mobile.symbols.aar'))
        stripped_aar_path = os.path.abspath(
                os.path.join(self._unpack_dir, 'maps.mobile.aar'))
        with zipfile.ZipFile(unstripped_aar_path, 'r') as zip:
            zip.extractall(path=self._unstripped_aar_dir)
        with zipfile.ZipFile(stripped_aar_path, 'r') as zip:
            zip.extractall(path=self._stripped_aar_dir)

        self._jdk_dir = os.path.join(self._unpack_dir, 'jdk')
        self._android_sdk_dir = os.path.join(self._unpack_dir, 'android_sdk')

        extract_resource(self.Parameters.jdk_resource, self._jdk_dir, gzip=False)
        extract_resource(self.Parameters.android_sdk_resource, self._android_sdk_dir)

    def _set_gradle_env(self):
        properties = {
            'libs_with_debug_info_path': os.path.join(self._unstripped_aar_dir, 'jni'),
            'stripped_libs_path': self._stripped_aar_dir,
            'keystorePath': 'null',
            'keystorePass': 'null',
            'keyAlias': 'null',
            'keyPass': 'null'
        }

        os.environ['USE_CRASHLYTICS'] = '1'
        self._gradle_env = os.environ
        self._gradle_env.update(
            {_GRADLE_PROPERTY_PREFIX + prop: value for (prop, value) in properties.items()})
        self._gradle_env.update({
            'JAVA_HOME': os.path.abspath(self._jdk_dir),
            'ANDROID_HOME': os.path.join(os.path.abspath(self._android_sdk_dir), 'android_sdk')
            })

    def _run_upload(self):
        gradlew_path = os.path.join(self._android_testapp_path, 'gradlew')
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('gradle_wrapper')) as pl:
            sdk2.helpers.subprocess.check_call(
                [gradlew_path,
                    '-d',
                    '--no-daemon',
                    'processReleaseGoogleServices',
                    'uploadCrashlyticsSymbolFileRelease'],
                cwd=self._android_testapp_path,
                env=self._gradle_env,
                stdout=pl.stdout,
                stderr=pl.stderr)
