import os
import requests
import tarfile

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.maps.common.retry import retry
from sandbox.common.errors import TaskError

BETA_ADMIN_URL = 'https://admin.beta.m.soft.yandex-team.ru'
BETA_UPLOAD_URL = os.path.join(BETA_ADMIN_URL, 'upload')
BETA_DB_PLATFORM_MAP = {
    'androidphone': 'android',
    'iphone': 'iphoneos'
    }


@retry(
    exceptions=requests.exceptions.RequestException,
    tries=5,
    delay=30,
    max_delay=120,
    backoff=2
    )
def _upload_app(full_path, params):
    with open(full_path, 'rb') as app:
        response = requests.post(
            BETA_UPLOAD_URL,
            params=params,
            files={"app_file": app}
            )
        response.raise_for_status()


@retry(
    exceptions=requests.exceptions.RequestException,
    tries=5,
    delay=30,
    max_delay=120,
    backoff=2
    )
def _get_branch_info(app, branch, platform):
    response = requests.get(
        os.path.join(BETA_ADMIN_URL, 'api', 'v1', app, branch, BETA_DB_PLATFORM_MAP[platform], ''))
    if response.status_code == 404:
        return None
    response.raise_for_status()
    return response.json()


class MapsMobileUploadToBeta(sdk2.Task):
    ''' Task for uploading mobile apps to beta '''

    class Parameters(sdk2.Task.Parameters):
        artifact_resource = sdk2.parameters.Resource('Artifact resource', required=True)
        artifact_path = sdk2.parameters.String('Path to artifact insinde resource', required=True)
        beta_platform = sdk2.parameters.String('Beta platform', required=True)
        beta_prefix = sdk2.parameters.String('Beta prefix', required=True)
        beta_bundle_id = sdk2.parameters.String('Beta bundle id', required=False)
        beta_branch = sdk2.parameters.String('Beta branch', required=True)

    def on_execute(self):
        branch_info = _get_branch_info(
            self.Parameters.beta_prefix,
            self.Parameters.beta_branch,
            self.Parameters.beta_platform
            )
        builds = branch_info['builds'] if branch_info else None
        latest_version = builds[0]['build_number'] if builds else None
        if latest_version and int(self.Parameters.artifact_resource.version) < int(latest_version):
            raise TaskError("Artifact version is older than beta latest build")

        extract_dir = 'extract'
        os.makedirs(extract_dir)
        tgz_path = str(sdk2.ResourceData(self.Parameters.artifact_resource).path)
        with tarfile.open(tgz_path, 'r:gz') as tgz:
            tgz.extractall(extract_dir)
        full_path = os.path.join(extract_dir, self.Parameters.artifact_path)
        params = {
            'platform': self.Parameters.beta_platform,
            'app': self.Parameters.beta_prefix,
            'branch': self.Parameters.beta_branch,
            'version': self.Parameters.artifact_resource.version,
            'build': self.Parameters.artifact_resource.version,
            'bundle_id': self.Parameters.beta_bundle_id
            }

        _upload_app(full_path, params)
