import re

from sandbox.common.errors import TaskError

_BRANCH_DATE_REGEX = re.compile('^arcadia:/arc/(branches/maps-mobile-releases/)?([^/]*)/arcadia(@\w+)?$')
_ARC_HASH_REGEX = re.compile('^arcadia-arc:/#\w+$')

_FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK = '2022012517'

_FIRST_BRANCH_WITH_M1_SIMULATOR_BUILD = '2022020209'


def extract_branch_name(arcadia_url):
    if _ARC_HASH_REGEX.match(arcadia_url):
        raise TaskError('Can\'t extract branch name from arc hash!')
    match = _BRANCH_DATE_REGEX.match(arcadia_url)
    assert match
    return match.group(2)


def is_branch_up_to_date(arcadia_url, first_branch_with_feature_presence):
    try:
        branch_name = extract_branch_name(arcadia_url)
    # we are not able to compare arc-hashes, lets consider arc branches always not up to date
    except TaskError:
        return False
    return branch_name >= first_branch_with_feature_presence or branch_name == 'trunk'


def validate_has_exact_svn_revision(arcadia_url):
    if not arcadia_url.startswith('arcadia:/') or '@' not in arcadia_url:
        raise TaskError('Only ArcadiaUrl conforming following format'
                          ' are supported: "arcadia:/arc/{branch}/arcadia@{revision}"')


class BranchDependentValue:
    def __init__(self, variants, default_value):
        self._variants = sorted(variants.iteritems(), key=lambda variant: variant[0], reverse=True)
        self._default_value = default_value

    def get(self, arcadia_url):
        for branch, value in self._variants:
            if is_branch_up_to_date(arcadia_url, branch):
                return value
        return self._default_value
