import logging

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
import sandbox.common.types.task as task_types
from sandbox.projects.maps.mobile.utils.subprocess_helpers import process_subtask_status


class SubtaskRunner():
    """
        Runs and waits for a subtask.
        To run two or more tasks of the same type using this SubtaskRunner, you MUST provide a unique tag for each one.
    """

    def __init__(self, parent_task):
        self._parent_task = parent_task

    # TODO: revert require_successful addition after MAPSMOBCORE-12327 is concluded
    def run_and_handle_result(self, subtask_type, subtask_parameters, result_callback=None,
                              description=None, tag=None, require_successful=True, subtask_requirements=None):
        if not tag:
            tag = subtask_type
        with self._parent_task.memoize_stage['run_{}'.format(tag)]:
            self.run(subtask_type, subtask_parameters, subtask_requirements, description=description, tag=tag)
        with self._parent_task.memoize_stage['handle_{}_result'.format(tag)]:
            subtask = self.find_task(tag)
            if require_successful:
                self.require_successful(subtask)
            if subtask.status == task_types.Status.SUCCESS and result_callback:
                result_callback(subtask)

    @staticmethod
    def wait_all(subtasks):
        raise sdk2.WaitTask(subtasks,
                            [task_types.Status.Group.FINISH, task_types.Status.Group.BREAK],
                            wait_all=True)

    def run(self, type, parameters, requirements=None, description=None, tag=None, wait=True):
        if not description:
            description = 'Subtask of {parent_type}'.format(parent_type=self._parent_task.type)
        if not tag:
            tag = type
        subtask = sdk2.Task[type](
            self._parent_task,
            description=description,
            inherit_notifications=True,
            **parameters
        )
        if requirements:
            for key in requirements:
                setattr(subtask.Requirements, key, requirements[key])
            subtask.save()
        subtask.enqueue()
        if 'subtask_ids' not in self._parent_task.Context:
            self._parent_task.Context.subtask_ids = {}
        self._parent_task.Context.subtask_ids[tag] = subtask.id
        if wait:
            self.wait_all([subtask])

    def find_task(self, tag):
        task_id = self._parent_task.Context.subtask_ids[tag]
        return sdk2.Task[task_id]

    def find_tasks(self, tags=None):
        return [sdk2.Task[task_id]
                for tag, task_id in self._parent_task.Context.subtask_ids.items()
                if not tags or tag in tags]

    @staticmethod
    def require_successful(subtask):
        logging.info('The {} subtask has finished with status {}'.format(subtask.type, subtask.status))
        process_subtask_status(subtask.status, 'Failed to run the {} subtask {}'.format(subtask.type, subtask.id))

    @staticmethod
    def require_all_successful(subtasks):
        for subtask in subtasks:
            logging.info('The {} subtask has finished with status {}'.format(subtask.type, subtask.status))

        error_msg = lambda failed_subtasks: 'Failed to run the subtasks:\n' + ';\n'.join('Type: {}, id: {}'.format(subtask.type, subtask.id)
                                                        for subtask in failed_subtasks)

        failing_subtasks = [subtask for subtask in subtasks if subtask.status == task_types.Status.FAILURE]
        if failing_subtasks:
            raise TaskFailure(error_msg(failing_subtasks))

        subtask_with_exception = [subtask for subtask in subtasks if subtask.status != task_types.Status.SUCCESS]
        if subtask_with_exception:
            raise TaskError(error_msg(subtask_with_exception))
