import datetime
import logging
import os
import tempfile
import tarfile
import shutil

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.arcadia import sdk as arcadiasdk

META_OPENEMBEDDED_DEFAULT_RESOURCE_ID = 1511301082
META_RAUC_DEFAULT_RESOURCE_ID         = 1511304275
META_ROCKCHIP_DEFAULT_RESOURCE_ID     = 1511307420
POKY_DEFAULT_RESOURCE_ID              = 1511309508

APP_SLUG = 'maps-mrc-drive'
MANIFEST_NAME = "manifest.raucm"
MANIFEST_TEMPLATE = """
[update]
compatible=nanopi
version={}

[image.rootfs]
filename={}
"""


class MapsMrcDriveNanopiImage(sdk2.Resource):
    """nanopi-neo4 rootfs image"""

    version = sdk2.parameters.String('Version of the bundle')


class MapsMrcDriveNanopiImageBuildTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        client_tags = (Tag.LINUX_XENIAL | Tag.LINUX_BIONIC)
        dns = DnsType.DNS64
        disk_space = 50 * 1024 # 50 Gb
        cores = 8
        ram = 4 * 1024 # 4 Gb

        container_resource = 1647964418

        environments = [
            sdk2.environments.PipEnvironment("yandex-signer-client",
               index_url="git+https://github.yandex-team.ru/idm/yandex-signer-client/")
        ]


    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 12 * 3600 # 12 hours

        with sdk2.parameters.RadioGroup('Firmware staging', required = True) as staging:
            staging.values.testing = None
            staging.values.stable = None

        layers_path = sdk2.parameters.String(
             'Path to yocto layers in Arcadia',
             required = True,
             default = 'maps/wikimap/mapspro/services/mrc/drive/mobile/yocto'
        )

        revision = sdk2.parameters.Integer('Arcadia revision', required = True)

        yav_secret = sdk2.parameters.YavSecret(
            'Yav secret with rauc bundle signing token and passwords',
            default = 'sec-01e90f7j14f3tdmeec9f0j7jen',
            required = True
        )

        yav_config_secret = sdk2.parameters.YavSecret(
            'Yav secret with X-YFirmware-Signature key',
            default = 'sec-01eac3jcxkchnkrjcsmes9eyjt',
            required = True
        )

        with sdk2.parameters.Group('3rd party yocto layers') as yocto_layers_3p:
            meta_openembedded_layer = sdk2.parameters.Resource(
                'meta-openembedded layer',
                default_value = META_OPENEMBEDDED_DEFAULT_RESOURCE_ID,
                required = True
            )

            meta_rauc_layer = sdk2.parameters.Resource(
                'meta-rauc layer',
                default_value = META_RAUC_DEFAULT_RESOURCE_ID,
                required = True
            )

            meta_rockchip_layer = sdk2.parameters.Resource(
                'meta-rockchip layer',
                default_value = META_ROCKCHIP_DEFAULT_RESOURCE_ID,
                required = True
            )

            poky_layer = sdk2.parameters.Resource(
                'poky layer',
                default_value = POKY_DEFAULT_RESOURCE_ID,
                required = True
            )


    def _prepare_layers(self, workdir):
        sources_path = os.path.join(workdir, 'sources')

        # Copy arcadia layers to the working dir
        with arcadiasdk.mount_arc_path(Arcadia.ARCADIA_TRUNK_URL + '@{}'.format(self.Parameters.revision)) as arcadia:
            arcadia_path = os.path.join(arcadia, self.Parameters.layers_path)
            shutil.copytree(arcadia_path, sources_path)

        # Copy 3rd party layers to the working dir
        layers_3p = [
            self.Parameters.meta_openembedded_layer,
            self.Parameters.meta_rauc_layer,
            self.Parameters.meta_rockchip_layer,
            self.Parameters.poky_layer
        ]

        for resource in layers_3p:
            layer_path = str(sdk2.ResourceData(resource).path)
            tarfile.open(layer_path).extractall(sources_path)


        with ProcessLog(logger=logging.getLogger('make_env_symlink')) as pl:
            command = ['ln', '-s',
                       os.path.join(sources_path, 'scripts', 'setup-environment.sh'),
                       os.path.join(workdir, 'setup-environment.sh')]
            sp.check_call(command, env=os.environ, stdout=pl.stdout, stderr=pl.stdout)


    def _build_image(self, workdir):
        """Build image with bitbake and return path to rootfs ext4 image"""

        staging = self.Parameters.staging

        with ProcessLog(logger=logging.getLogger('bitbake')) as pl, arcadiasdk.mount_arc_path(Arcadia.ARCADIA_TRUNK_URL + '@{}'.format(self.Parameters.revision)) as arcadia:
            secret = self.Parameters.yav_secret.data()
            config_secret = self.Parameters.yav_config_secret.data()

            os.environ["ARCADIA"] = arcadia
            os.environ["DISTRO"] = "rk-none"
            os.environ["MACHINE"] = "nanopi-neo4"
            os.environ["PW_TESTING_MRC"] = str(secret["TESTING_MRC"])
            os.environ["PW_TESTING_ROOT"] = str(secret["TESTING_ROOT"])
            os.environ["PW_STABLE_MRC"] = str(secret["STABLE_MRC"])
            os.environ["PW_STABLE_ROOT"] = str(secret["STABLE_ROOT"])
            os.environ["SIGNATURE_KEY"] = str(config_secret["SIGNATURE_KEY"])

            command = ('source ./setup-environment.sh build && bitbake rk-image-drive-{}'.format(staging))

            logging.info('Run command: {}'.format(command))

            sp.check_call(command, env=os.environ,
                          stdout=pl.stdout, stderr=pl.stdout,
                          shell=True, executable='/bin/bash')

            logging.info('Bitbake finished')

        with ProcessLog(logger=logging.getLogger('disk_usage')) as pl:
            sp.check_call(['du', '-sh', workdir], stdout=pl.stdout, stderr=pl.stdout)

        return os.path.join(workdir, 'build', 'tmp', 'deploy', 'images', 'nanopi-neo4',
                            'rk-image-drive-{}-nanopi-neo4.ext4'.format(staging))


    def _make_version(self):
        d = datetime.date.today()
        return '{}.{}.{}-{}'.format(d.year, d.month, d.day, self.Parameters.revision)


    def _create_manifest(self, dir, version, image_filename):
        manifest_path = os.path.join(dir, MANIFEST_NAME)
        with open(manifest_path, 'w') as manifest:
            manifest.write(MANIFEST_TEMPLATE.format(version, image_filename))


    def _create_rauc_bundle(self, image_path, version, output_path):
        logging.info('Create rauc bundle from image {}'.format(image_path))

        input_dir = 'rauc_input'
        os.makedirs(input_dir)
        image_filename = 'rootfs.img.ext4'
        shutil.copyfile(image_path, os.path.join(input_dir, image_filename))
        self._create_manifest(input_dir, version, image_filename)

        signing_input_path = input_dir + ".tar.gz"
        with tarfile.open(signing_input_path, "w:gz") as tar:
            for name in [image_filename, MANIFEST_NAME]:
                tar.add(os.path.join(input_dir, name), arcname=name)

        secret = self.Parameters.yav_secret.data()
        oauth_token = secret['SIGNER_OAUTH_TOKEN']

        with ProcessLog(logger=logging.getLogger('sign_rauc_bundle')) as pl:
            command = ['ya-signer',
                       '--token', oauth_token,
                       '--app-slug', APP_SLUG,
                       '--infile', signing_input_path,
                       '--outfile', output_path,
                       '--request_timeout', '600',
                       '--comment', 'Sign version {} in sandbox'.format(version)]
            sp.check_call(command, stdout=pl.stdout, stderr=pl.stdout)


    def _publish_resource(self, publish_path, version):
        logging.info('Publish rauc bundle as a resource')

        image_resource = MapsMrcDriveNanopiImage(self, 'rootfs image bundle', publish_path)
        image_resource.version = version
        image_data = sdk2.ResourceData(image_resource)
        image_data.ready()


    def on_execute(self):
        cwd = os.getcwd()
        logging.info('Current dir: {}'.format(cwd))

        workdir = tempfile.mkdtemp()
        logging.info('Work dir: {}'.format(workdir))

        try:
            self._prepare_layers(workdir)

            os.chdir(workdir)
            image_path = self._build_image(workdir)
            os.chdir(cwd)

            bundle_path = os.path.join(cwd, 'rootfs-nanopi-neo4.raucb')
            version = self._make_version()

            self._create_rauc_bundle(image_path, version, bundle_path)
            self._publish_resource(bundle_path, version)
        finally:
            shutil.rmtree(workdir)
