import logging
import os
import tempfile
import tarfile
import shutil

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.arcadia import sdk as arcadiasdk

TAXI_REPO_URL_TEMPLATE = 'https://x-oauth-token:{}@bb.yandex-team.ru/scm/taxi/signalq2.git'
FW_UPDATER_PRODUCTION_SECRET_ID = 'sec-01eac3jcxkchnkrjcsmes9eyjt'
FW_UPDATER_TESTING_SECRET_ID = 'sec-01eac3cbqbxf79k7ynxt9zzzhn'
MRC_DRIVE_FIRMWARE_SECRET_ID = 'sec-01e90f7j14f3tdmeec9f0j7jen'

TAXI_REPO_CHECKOUT_DIR = 'signalq2'
BUILD_DIR = 'sdk/ambalink_sdk_4_14/output.oem/ambalink/images/'

class MapsMrcDriveSignalQ2Image(sdk2.Resource):
    """SignalQ2 firmware"""

    version = sdk2.parameters.String('Firmware version')
    staging = sdk2.parameters.String('Firmware staging')
    hw_version = sdk2.parameters.String('Hardware version')
    update_type = sdk2.parameters.String('Update type')
    is_gpsfake_build = sdk2.parameters.Bool('Built with gpsfake')


class MapsMrcDriveSignalQ2Error(sdk2.Resource):
    """SignalQ2 error build data"""

    hw_version = sdk2.parameters.String('Hardware version')


class MapsMrcDriveSignalQ2ImageBuildTask(sdk2.Task):

    class Requirements(sdk2.Requirements):
        client_tags = (Tag.LINUX_XENIAL | Tag.LINUX_BIONIC)
        dns = DnsType.DNS64
        cores = 1
        disk_space = 40 * 1024 # 40 Gb
        ram = 4 * 1024 # 4 Gb

        container_resource = 1883774193

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 6 * 3600 # 6 hours

        with sdk2.parameters.RadioGroup('Firmware updater environment', required = True) as fw_updater_env:
            fw_updater_env.values.production = fw_updater_env.Value(value = 'production', default = True)
            fw_updater_env.values.testing = fw_updater_env.Value(value = 'testing')

        with sdk2.parameters.RadioGroup('SignalQ2 hardware id', required = True) as hw_version:
            hw_version.values['2.1-dev'] = hw_version.Value(value = '2.1-dev')
            hw_version.values['2.1-mp'] = hw_version.Value(value = '2.1-mp')

        with sdk2.parameters.RadioGroup('Firmware staging', required = True) as staging:
            staging.values.production = None
            staging.values.testing = None

        revision = sdk2.parameters.Integer('Arcadia revision', required = True)

        gpsfake_nmea = sdk2.parameters.Resource('NMEA resource (for gpsfake build)', required = False)

        mrc_drive_firmware_yav_secret = sdk2.parameters.YavSecret(
            'Yav secret with bitbucket and arc tokens',
            default = MRC_DRIVE_FIRMWARE_SECRET_ID,
            required = True
        )


    def _checkout_repo(self, workdir):
        """Clone taxi repository and checkout mrc-master branch"""

        cwd = os.getcwd()
        os.chdir(workdir)

        secret = self.Parameters.mrc_drive_firmware_yav_secret.data()
        oauth_token = secret['BITBUCKET_OAUTH_TOKEN']
        repo_url = TAXI_REPO_URL_TEMPLATE.format(oauth_token)

        logging.info('Checking out taxi repo...')
        with ProcessLog(logger=logging.getLogger('checkout_repo')) as pl:
            command = ['git', 'clone', '--branch', 'mrc-master', repo_url, TAXI_REPO_CHECKOUT_DIR]
            sp.check_call(command, env=os.environ, stdout=pl.stdout, stderr=pl.stdout)

            os.chdir(TAXI_REPO_CHECKOUT_DIR)

            logging.info('Install git LFS')
            sp.check_call(['git', 'lfs', 'install'], env=os.environ, stdout=pl.stdout, stderr=pl.stdout)
            sp.check_call(['git', 'lfs', 'pull'], env=os.environ, stdout=pl.stdout, stderr=pl.stdout)

            logging.info('Create fake visage license file')
            with open(os.path.join('signalq', 'visage_license.vlc'), 'w') as visage_lic:
                visage_lic.write('ok')

        os.chdir(cwd)


    def _get_firmware_version(self, workdir):
        linux_release_path = os.path.join(workdir, TAXI_REPO_CHECKOUT_DIR, 'release')
        mrc_release_path = os.path.join(workdir, TAXI_REPO_CHECKOUT_DIR, 'mrc-release')

        with open(linux_release_path) as file:
            linux_version_full = file.read().strip()

        with open(mrc_release_path) as file:
            mrc_version = file.read().strip()

        linux_version_short = linux_version_full.split('_')[0]
        return linux_version_short + '_' + mrc_version


    def _build_image(self, workdir):
        """Build SignalQ2 firmware and return path to resulting directory"""

        staging = self.Parameters.staging
        hw_version = self.Parameters.hw_version
        fw_updater_env = self.Parameters.fw_updater_env
        yav_config_secret_id = FW_UPDATER_TESTING_SECRET_ID if fw_updater_env == 'testing' else FW_UPDATER_PRODUCTION_SECRET_ID
        gpsfake_nmea_id = self.Parameters.gpsfake_nmea.id if self.Parameters.gpsfake_nmea else None

        arcadia_url = Arcadia.ARCADIA_TRUNK_URL
        if self.Parameters.revision > 0:
            arcadia_url += '@{}'.format(self.Parameters.revision)

        cwd = os.getcwd()

        with ProcessLog(logger=logging.getLogger('firmware-build')) as pl, arcadiasdk.mount_arc_path(arcadia_url) as arcadia:
            os.environ["ARCADIA"] = arcadia
            os.environ["BUILD_REVISION"] = hw_version
            os.environ["MRC_CONFIG_YAV_SECRET"] = yav_config_secret_id
            os.environ["YAV_TOKEN"] = self.Parameters.mrc_drive_firmware_yav_secret.data()['YAV_OAUTH_TOKEN']
            os.environ["STAGING"] = staging
            if gpsfake_nmea_id:
                os.environ["NMEA_RESOURCE_ID"] = str(gpsfake_nmea_id)

            os.chdir(os.path.join(workdir, TAXI_REPO_CHECKOUT_DIR))

            command = ('./local.sh')
            logging.info('Start building firmware with command: {}'.format(command))
            sp.check_call(command, env=os.environ,
                          stdout=pl.stdout, stderr=pl.stdout,
                          shell=True, executable='/bin/bash')

            logging.info('Build finished')

        # Build result should appear in ./${BUILD_DIR}/<hw_version>-... folder
        subdir = [dir for dir in os.listdir(BUILD_DIR) if dir.startswith(hw_version)][0]

        os.chdir(cwd)

        return os.path.join(workdir,
                            TAXI_REPO_CHECKOUT_DIR,
                            BUILD_DIR,
                            subdir)


    def _publish_resource(self, publish_path, version, update_type):
        logging.info('Publish firmware archive as a resource')

        image_resource = MapsMrcDriveSignalQ2Image(self, 'SignalQ2 firmware', publish_path)
        image_resource.version = version
        image_resource.staging = self.Parameters.staging
        image_resource.hw_version = self.Parameters.hw_version
        image_resource.update_type = update_type
        image_resource.is_gpsfake_build = True if self.Parameters.gpsfake_nmea else False

        image_data = sdk2.ResourceData(image_resource)
        image_data.ready()


    def _get_ota_firmware_path(self, dir):
        for file in os.listdir(dir):
            if file.startswith('linux') and file.endswith('.tar.gz'):
                return os.path.join(dir, file)
        raise TaskFailure('Cannot find tar.gz firmware among build results')


    def _get_usb_firmware_path(self, dir):
        for file in os.listdir(dir):
            if file.endswith('.elf'):
                return os.path.join(dir, file)
        raise TaskFailure('Cannot find elf firmware among build results')


    def on_execute(self):
        cwd = os.getcwd()
        logging.info('Current dir: {}'.format(cwd))

        workdir = tempfile.mkdtemp()
        logging.info('Work dir: {}'.format(workdir))

        try:
            self._checkout_repo(workdir)

            results_dir = self._build_image(workdir)
            version = self._get_firmware_version(workdir)

            ota_firmware_build_path = self._get_ota_firmware_path(results_dir)
            ota_firmware_publish_path = os.path.join(cwd, 'signalq2-firmware.tar.gz')
            shutil.copyfile(ota_firmware_build_path, ota_firmware_publish_path)
            self._publish_resource(ota_firmware_publish_path, version, 'OTA')

            usb_firmware_build_path = self._get_usb_firmware_path(results_dir)
            usb_firmware_publish_path = os.path.join(cwd, 'signalq2-firmware.elf')
            shutil.copyfile(usb_firmware_build_path, usb_firmware_publish_path)
            self._publish_resource(usb_firmware_publish_path, version, 'USB')

        finally:
            shutil.rmtree(workdir)
