import datetime
import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.org_status_pushes.common.resources import (
    SendPushesExecutable
)
from sandbox.projects.maps.org_status_pushes.common.utils import (
    load_executable_resource
)


class PrepareOrganizationsExecutable(sdk2.Resource):
    resource_name = 'prepare_organizations'
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PreparePushesExecutable(sdk2.Resource):
    resource_name = 'prepare_pushes'
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ProcessPushesExecutable(sdk2.Resource):
    resource_name = 'process_pushes'
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class BuildOrgStatusPushes(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB
        environments = [environments.PipEnvironment('yandex-yt')]

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        owner = 'MAPS'
        kill_timeout = 12 * 24 * 60 * 60  # 12 hours

        with sdk2.parameters.Group('Executable resources') as executable_resources:
            prepare_organizations_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of prepare_organizations executable',
                resource_type=PrepareOrganizationsExecutable,
                required=False
            )
            prepare_pushes_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of prepare_pushes executable',
                resource_type=PreparePushesExecutable,
                required=False
            )
            process_pushes_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of process_pushes executable',
                resource_type=ProcessPushesExecutable,
                required=False
            )
            send_pushes_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of send_pushes executable',
                resource_type=SendPushesExecutable,
                required=False
            )

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        with sdk2.parameters.Group('Pushes parameters') as pushes_parameters:
            pushes_export_dir = sdk2.parameters.String(
                'YT-directory with pushes export',
                default='//home/maps/poi/notification/geoplatform_org_status'
            )
            permalink_table = sdk2.parameters.String('The table with organizations permalinks')
            days = sdk2.parameters.Integer(
                'Number of days for which information about user activity is retrieved',
                default=7
            )
            max_distance_meters = sdk2.parameters.Float(
                'Maximum distance from user to organization for sending a push', default=100.0
            )
            min_events_per_user_org = sdk2.parameters.Integer(
                'Minimum metrika events per user-organization pair', default=10
            )
            max_user_orgs_count = sdk2.parameters.Integer(
                'Maximum number of organizations per user', default=10
            )
            actualized_organization_cooldown_days = sdk2.parameters.Integer(
                'Number of days while actualized organizations are not included', default=30
            )
            user_cooldown_days = sdk2.parameters.Integer(
                'Number of days while it is not allowed to send pushes to the user'
            )
            user_organization_cooldown_days = sdk2.parameters.Integer(
                'Number of days while it is not allowed to send pushes to the user '
                'for the particular organization'
            )
            user_useless_cooldown_days = sdk2.parameters.Integer(
                'Number of days while it is not allowed to send pushes to the user '
                'after clicking useless button'
            )
            user_not_opened_cooldown_days = sdk2.parameters.Integer(
                'Number of days while it is not allowed to send pushes to the user'
                'after not opening several pushes'
            )
            user_not_opened_count = sdk2.parameters.Integer(
                'Number of not opened pushes for --user-not-opened-cooldown-days'
            )
            experiments = sdk2.parameters.String(
                'List of experiment tags with users percent to apply'
            )
            with sdk2.parameters.RadioGroup('Receiver') as receiver:
                receiver.values.uuid = receiver.Value(value='uuid', default=True)
                receiver.values.puid = receiver.Value(value='puid')
            with sdk2.parameters.RadioGroup('Target') as target:
                target.values.feedback_form = target.Value(value='feedback_form', default=True)
                target.values.ugc_account = target.Value(value='ugc_account')

    def _copy_yt_table(self, source, dest):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()
        yt.copy(source, dest)

    def _create_export_dir(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        utc_date = datetime.datetime.utcnow().isoformat()
        export_dir = yt.ypath_join(self.Parameters.pushes_export_dir, utc_date)
        yt.create('map_node', export_dir)
        self.Context.export_dir = export_dir

    def ensure_latest_resources_used(self):
        self.prepare_organizations_executable = self.Parameters.prepare_organizations_executable
        if not self.prepare_organizations_executable:
            self.prepare_organizations_executable = find_latest_resource(
                PrepareOrganizationsExecutable, self.Parameters.environment)

        self.prepare_pushes_executable = self.Parameters.prepare_pushes_executable
        if not self.prepare_pushes_executable:
            self.prepare_pushes_executable = find_latest_resource(
                PreparePushesExecutable, self.Parameters.environment)

        self.process_pushes_executable = self.Parameters.process_pushes_executable
        if not self.process_pushes_executable:
            self.process_pushes_executable = find_latest_resource(
                ProcessPushesExecutable, self.Parameters.environment)

        self.send_pushes_executable = self.Parameters.send_pushes_executable
        if not self.send_pushes_executable:
            self.send_pushes_executable = find_latest_resource(
                SendPushesExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using PrepareOrganizationsExecutable: %s',
            self.prepare_organizations_executable.id)
        logging.info(
            'Using PreparePushesExecutable: %s',
            self.prepare_pushes_executable.id)
        logging.info(
            'Using ProcessPushesExecutable: %s',
            self.process_pushes_executable.id)
        logging.info(
            'Using SendPushesExecutable: %s',
            self.send_pushes_executable.id)

    def on_execute(self):
        self.ensure_latest_resources_used()

        load_executable_resource(self.prepare_organizations_executable)
        load_executable_resource(self.prepare_pushes_executable)
        load_executable_resource(self.process_pushes_executable)
        load_executable_resource(self.send_pushes_executable)

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()
        self._create_export_dir()

        with self.memoize_stage.prepare_organizations:
            if not self.Parameters.permalink_table:
                cmd = [
                    './prepare_organizations',
                    '--actualized-organization-cooldown-days',
                    str(self.Parameters.actualized_organization_cooldown_days),
                    '--output-table', os.path.join(self.Context.export_dir, 'permalinks')
                ]
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('prepare organizations')) as pl:
                    subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
            else:
                self._copy_yt_table(
                    self.Parameters.permalink_table,
                    os.path.join(self.Context.export_dir, 'permalinks')
                )

        with self.memoize_stage.prepare_pushes:
            cmd = [
                './prepare_pushes',
                '--permalink-table', os.path.join(self.Context.export_dir, 'permalinks'),
                '--days', str(self.Parameters.days),
                '--max-user-org-distance-meters', str(self.Parameters.max_distance_meters),
                '--min-events-per-user-org', str(self.Parameters.min_events_per_user_org),
                '--max-user-orgs-count', str(self.Parameters.max_user_orgs_count),
                '--output-dir', self.Context.export_dir
            ]
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('prepare pushes')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        with self.memoize_stage.process_pushes:
            cmd = [
                './process_pushes',
                '--input-dir', self.Context.export_dir,
                '--pushes-export-dir', self.Parameters.pushes_export_dir
            ]
            if self.Parameters.user_cooldown_days:
                cmd += ['--user-cooldown-days', str(self.Parameters.user_cooldown_days)]
            if self.Parameters.user_organization_cooldown_days:
                cmd += ['--user-organization-cooldown-days', str(self.Parameters.user_organization_cooldown_days)]
            if self.Parameters.user_useless_cooldown_days:
                cmd += ['--user-useless-cooldown-days', str(self.Parameters.user_useless_cooldown_days)]
            if self.Parameters.user_not_opened_cooldown_days:
                cmd += ['--user-not-opened-cooldown-days', str(self.Parameters.user_not_opened_cooldown_days)]
            if self.Parameters.user_not_opened_count:
                cmd += ['--user-not-opened-count', str(self.Parameters.user_not_opened_count)]
            if self.Parameters.experiments:
                cmd += ['--experiments']
                cmd += str(self.Parameters.experiments).split(' ')

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('process pushes')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        with self.memoize_stage.pack_pushes:
            cmd = [
                './send_pushes',
                '--pack-only',
                '--input-tables', os.path.join(self.Context.export_dir, 'pushes_processed'),
                '--data-table', os.path.join(self.Context.export_dir, 'pushes_data'),
                '--receiver', self.Parameters.receiver,
                '--target', self.Parameters.target,
            ]
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('pack pushes')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
