import logging
import os
import subprocess

from sandbox import sdk2

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.org_status_pushes.common.utils import (
    load_executable_resource
)


class CalculateStatisticsExecutable(sdk2.Resource):
    resource_name = 'calculate_statistics'
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class CalculateOrgStatusPushesStatistics(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        owner = 'MAPS'
        kill_timeout = 3 * 24 * 60 * 60  # 3 hours

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        with sdk2.parameters.Group('Executable resources') as executable_resources:
            calculate_statistics_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of calculate_statistics executable',
                resource_type=CalculateStatisticsExecutable,
                required=False
            )

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            statistics_table = sdk2.parameters.String(
                'YT-table with pushes statistics',
                default='//home/maps/poi/notification/statistics/geoplatform_org_status'
            )

    def ensure_latest_resources_used(self):
        self.calculate_statistics_executable = self.Parameters.calculate_statistics_executable
        if not self.calculate_statistics_executable:
            self.calculate_statistics_executable = find_latest_resource(
                CalculateStatisticsExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using CalculateStatisticsExecutable: %s',
            self.calculate_statistics_executable)

    def on_execute(self):
        self.ensure_latest_resources_used()

        load_executable_resource(self.calculate_statistics_executable)
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()

        with self.memoize_stage.calculate_statistics:
            cmd = [
                './calculate_statistics',
                '--statistics-table', self.Parameters.statistics_table
            ]
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('calculate_statistics')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
