import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk import environments

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)


TVM_SECRET_YAV_ID_TESTING = 'sec-01eynrac9jcfqwjqkvh6vh8c34'
TVM_SECRET_YAV_ID_STABLE = 'sec-01eynrfrq9j0yd556fxbgzqeak'


class CreateUgcAccountTasksExecutable(sdk2.Resource):
    resource_name = 'create_ugc_account_tasks'
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class CreateUgcAccountTasks(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        cores = 16
        ram = 32 * 1024  # 32 GB
        environments = [environments.PipEnvironment('yandex-yt')]

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        kill_timeout = 48 * 60 * 60  # 48 hours

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)

        with sdk2.parameters.Group('Executable resources') as executable_resources:
            create_ugc_account_tasks_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of create_ugc_account_tasks executable',
                resource_type=CreateUgcAccountTasksExecutable,
                required=False
            )

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            export_dir = sdk2.parameters.String(
                'YT-directory on hahn with list of folders with tasks data',
                default='//home/maps/poi/notification/geoplatform_org_status'
            )
            export_date = sdk2.parameters.String('YT-directory in tasks export')
            export_table_name = sdk2.parameters.String(
                'YT-table name in date dirictory',
                default='pushes_processed'
            )
            n_jobs = sdk2.parameters.Integer(
                'Number of jobs to run in parallel',
                default=4
            )
            log = sdk2.parameters.String(
                'logging level',
                default='ERROR'
            )
            with sdk2.parameters.RadioGroup('Task type') as task_type:
                task_type.values.org_status = task_type.Value(value='org_status', default=True)
                task_type.values.address_add = task_type.Value(value='address_add')
                task_type.values.barriers = task_type.Value(value='barriers')
                task_type.values.gates = task_type.Value(value='gates')

            with sdk2.parameters.RadioGroup('Environment to create tasks') as environment:
                environment.values.testing = environment.Value(value='testing', default=True)
                environment.values.stable = environment.Value(value='stable')

    def _get_latest_exported_date(self):
        import yt.wrapper as yt
        latest_date = sorted(yt.list(self.Parameters.export_dir, absolute=False))[-1]
        sup_logs = yt.ypath_join(self.Parameters.export_dir, latest_date, 'sup_logs')
        return None if yt.exists(sup_logs) else latest_date

    def ensure_latest_resources_used(self):
        self.create_ugc_account_tasks_executable = self.Parameters.create_ugc_account_tasks_executable
        if not self.create_ugc_account_tasks_executable:
            self.create_ugc_account_tasks_executable = find_latest_resource(
                CreateUgcAccountTasksExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using CreateUGCAccountTasksExecutable: %s',
            self.create_ugc_account_tasks_executable.id)

    def on_execute(self):
        import yt.wrapper as yt
        yt.config.set_proxy('hahn')
        yt.config['token'] = self.Parameters.yt_vault.data()
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()

        tvm_secret_id = (TVM_SECRET_YAV_ID_STABLE
                         if self.Parameters.environment == 'stable'
                         else TVM_SECRET_YAV_ID_TESTING)
        os.environ['TVM_SECRET'] = sdk2.yav.Secret(tvm_secret_id).data()['client_secret']

        self.ensure_latest_resources_used()

        export_date = self.Parameters.export_date or self._get_latest_exported_date()
        if export_date is None:
            raise TaskError('Latest date from {} is already sent'.format(self.Parameters.export_dir))
        export_dir = yt.ypath_join(self.Parameters.export_dir, export_date)

        cmd = [
            str(sdk2.ResourceData(self.create_ugc_account_tasks_executable).path),
            '--input-table', yt.ypath_join(export_dir, self.Parameters.export_table_name),
            '--n-jobs', str(self.Parameters.n_jobs),
            '--log', self.Parameters.log,
            '--task-type', self.Parameters.task_type,
            '--env', self.Parameters.environment,
        ]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('create ugc account tasks')) as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
