import datetime
import logging
import os
import subprocess
import time

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk import environments

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.org_status_pushes.common.resources import (
    SendPushesExecutable
)
from sandbox.projects.maps.org_status_pushes.common.utils import (
    load_executable_resource
)


class SendOrgStatusPushes(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB
        environments = [environments.PipEnvironment('yandex-yt')]

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values['testing'] = environment.Value('testing', default=True)
            environment.values['stable'] = environment.Value('stable')

        owner = 'MAPS'
        kill_timeout = 8 * 24 * 60 * 60  # 8 hours, TODO: deal with more users

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            sup_vault = sdk2.parameters.Vault('SUP vault', required=False)
            logbroker_vault = sdk2.parameters.Vault('Logbroker vault', required=False)

        with sdk2.parameters.Group('Executable resources') as executable_resources:
            send_pushes_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of send_pushes executable',
                resource_type=SendPushesExecutable,
                required=False
            )

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            pushes_export_dir = sdk2.parameters.String(
                'YT-directory with pushes export',
                default='//home/maps/poi/notification/geoplatform_org_status'
            )
            export_date = sdk2.parameters.String('Date in pushes export')

        with sdk2.parameters.Group('Logbroker parameters') as logbroker_parameters:
            use_logbroker = sdk2.parameters.Bool('Use logbroker', default_value=False)
            logbroker_username = sdk2.parameters.String('Logbroker username', required=False)

    def _get_last_exported_date(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        pushes_export_dates = yt.list(self.Parameters.pushes_export_dir, absolute=False)
        if not pushes_export_dates:
            return None
        last_date = sorted(pushes_export_dates)[-1]
        pushes_data = os.path.join(self.Parameters.pushes_export_dir, last_date, 'pushes_data')
        sup_logs = os.path.join(self.Parameters.pushes_export_dir, last_date, 'sup_logs')
        if yt.exists(pushes_data) and not yt.exists(sup_logs):  # prepared but not sent
            return last_date
        else:
            return None

    def _save_send_date(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        utc_date = datetime.datetime.utcfromtimestamp(time.time()).strftime('%Y-%m-%dT%H:%M:%S.%fZ')
        send_date_table = yt.ypath_join(
            self.Context.export_dir, 'send_date_{}'.format(utc_date)
        )
        yt.create('table', send_date_table)

    def ensure_latest_resources_used(self):
        self.send_pushes_executable = self.Parameters.send_pushes_executable
        if not self.send_pushes_executable:
            self.send_pushes_executable = find_latest_resource(SendPushesExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using BuildStaticPoiExecutable: %s',
            self.send_pushes_executable.id)

    def on_execute(self):
        self.ensure_latest_resources_used()

        load_executable_resource(self.send_pushes_executable)
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        if self.Parameters.sup_vault:
            os.environ['SUP_TOKEN'] = self.Parameters.sup_vault.data()
        if self.Parameters.logbroker_vault:
            os.environ['LOGBROKER_TOKEN'] = self.Parameters.logbroker_vault.data()

        with self.memoize_stage.send_pushes:
            export_date = self.Parameters.export_date
            if not export_date:
                export_date = self._get_last_exported_date()
            if not export_date:
                raise TaskError('No data to send')
            export_dir = os.path.join(self.Parameters.pushes_export_dir, export_date)
            self.Context.export_dir = export_dir

            cmd = [
                './send_pushes',
                '--send-only',
                '--data-table', os.path.join(self.Context.export_dir, 'pushes_data'),
                '--sup-logs-table', os.path.join(self.Context.export_dir, 'sup_logs')
            ]

            if self.Parameters.use_logbroker:
                cmd.extend(['--logbroker-username', self.Parameters.logbroker_username])

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('send pushes')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        with self.memoize_stage.save_send_date:
            self._save_send_date()
