import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess


class BasemapPoiCheckerBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class BasemapPoiCheckerYqlUdf(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class BasemapPoiCheckerTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024  # 1 GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Executable') as executable_resources:
            checker_binary = sdk2.parameters.Resource(
                'Sandbox resource ID of checker executable',
                resource_type=BasemapPoiCheckerBinary,
                required=True
            )
            checker_yql_udf = sdk2.parameters.Resource(
                'Sandbox resource ID of checker .so yql_udf',
                resource_type=BasemapPoiCheckerYqlUdf,
                required=True
            )

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        with sdk2.parameters.Group('Checker parameters') as checker_parameters:
            permalinks_table = sdk2.parameters.String(
                'The table with organizations permalinks', required=True)
            pool = sdk2.parameters.String('YT pool to use', required=False)
            output_table = sdk2.parameters.String('YT output table', required=True)

    def _init_environment(self):
        os.environ['YT_PROXY'] = 'hahn'
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()
        os.environ['NILE_YQL_PYTHON_UDF_PATH'] = \
            str(sdk2.ResourceData(self.Parameters.checker_yql_udf).path)

    def on_execute(self):
        self._init_environment()

        cmd = [
            str(sdk2.ResourceData(self.Parameters.checker_binary).path),
            '--permalinks-table', self.Parameters.permalinks_table,
            '--output-table', self.Parameters.output_table
        ]
        if self.Parameters.pool:
            cmd += ['--pool', self.Parameters.pool]

        with ProcessLog(self, 'Executable') as pl:
            try:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
            # TODO: Call error is suppressed temporarly, just to allow perspoi
            # work as usual. Should be removed in https://st.yandex-team.ru/GEOQ-1003
            except subprocess.CalledProcessError as e:
                logging.warning(
                    'command "{}" failed with code {}: {}'.format(e.cmd, e.returncode, e.output)
                )
