import os

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqResourceWithAutoDiscovery


DATE_FORMAT = '%Y-%m-%d'


class PoiImpressionsMetricsExecutable(GeoqResourceWithAutoDiscovery):
    pass


class PoiImpressionsMetrics(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):
        with sdk2.parameters.Group('Executable'):
            executable = sdk2.parameters.Resource(
                'Impressions metrics executable (maps/poi/statistics/impressions/metrics)',
                resource_type=PoiImpressionsMetricsExecutable)

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_token = sdk2.parameters.YavSecret(
                'YT OAuth token', required=True)
            stat_token = sdk2.parameters.YavSecret(
                'Stat OAuth token', required=True)
            yt_pool = sdk2.parameters.String(
                'YT procesing pool',
                default='maps-core-poi-statistics')

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            orginfo_table = sdk2.parameters.String(
                'Orginfo table',
                default='//home/maps/poi/data/orginfo')
            org_statistics_table = sdk2.parameters.String(
                'Org statistics table',
                default='//home/maps/poi/data/org_statistics')
            renderer_impressions_table = sdk2.parameters.String(
                'Renderer impressions')

        with sdk2.parameters.Group('Pipeline parameters') as pipeline_parameters:
            min_zoom = sdk2.parameters.Integer(
                'Min impressions zoom', default=10)
            max_zoom = sdk2.parameters.Integer(
                'Min impressions zoom', default=21)

        with sdk2.parameters.Group('Output parameters') as output_parameters:
            output_folded_table = sdk2.parameters.String(
                'Folded table path')
            output_date = sdk2.parameters.String(
                'Force report date')

        with sdk2.parameters.Group('Statface parameters') as statface_parameters:
            statface_report_path = sdk2.parameters.String(
                'Statface report path')
            statface_report_title = sdk2.parameters.String(
                'Statface report title',
                default='Poi Impressions')
            with sdk2.parameters.RadioGroup('Statface proxy to use') as statface_proxy:
                statface_proxy.values.prod = statface_proxy.Value(
                    value='prod', default=True)
                statface_proxy.values.beta = statface_proxy.Value(
                    value='beta')

    def on_execute(self):
        date = self.Parameters.output_date or datetime.utcnow().strftime(DATE_FORMAT)

        os.environ.update({
            'YT_TOKEN': self.Parameters.yt_token.value(),
            'STAT_TOKEN': self.Parameters.stat_token.value()
        })

        arguments = [
            '--orginfo-table', self.Parameters.orginfo_table,
            '--org-statistics-table', self.Parameters.org_statistics_table,
            '--output-date', date,
        ]

        if self.Parameters.yt_pool:
            arguments += [
                '--yt-pool', self.Parameters.yt_pool,
            ]

        if self.Parameters.renderer_impressions_table:
            arguments += [
                '--renderer-impressions-table', self.Parameters.renderer_impressions_table
            ]

        if self.Parameters.output_folded_table:
            arguments += [
                '--output-folded-table', self.Parameters.output_folded_table
            ]

        if self.Parameters.statface_report_path:
            arguments += [
                '--statface-report-path', self.Parameters.statface_report_path,
                '--statface-report-title', self.Parameters.statface_report_title,
                '--statface-proxy', self.Parameters.statface_proxy
            ]

        self.run_executable(self.Parameters.executable, arguments)
