import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.common.errors import ResourceNotFound, TaskError
import sandbox.common.types.task as ctt

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)

import resources


class PoiMetricsExecutable(TaskJugglerReportWithParameters):
    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        with sdk2.parameters.Group('Sandbox Vault Secret') as yav_params:
            vault_owner = sdk2.parameters.String(
                'Sandbox Vault owner',
                required=True)
            yt_token_vault_name = sdk2.parameters.String(
                'YT token name',
                required=True,
                default='YT_TOKEN')
            yql_token_vault_name = sdk2.parameters.String(
                'YQL token name',
                required=True,
                default='YQL_TOKEN')
            statface_token_vault_name = sdk2.parameters.String(
                'Statface token name',
                required=True,
                default='STATBOX_TOKEN')

        with sdk2.parameters.Group('Main executable') as executable_params:
            main_executable = sdk2.parameters.Resource(
                'Main executable',
                resource_type=resources.PoiMetricsExecutableResource,
                required=False)
            main_args = sdk2.parameters.List(
                'Main executable arguments',
                default=[])

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_pool = sdk2.parameters.String(
                'YT procesing pool',
                default='maps-core-poi-statistics')

    def get_resource_path(self, resource_id):
        if resource_id is None:
            raise ResourceNotFound('Resource was not found')
        resource = sdk2.ResourceData(resource_id)
        return str(resource.path)

    def get_vault_secret(self, vault_name):
        return sdk2.Vault.data(self.Parameters.vault_owner, vault_name)

    def prepare_env(self):
        variables = {
            'YT_TOKEN': self.get_vault_secret(
                self.Parameters.yt_token_vault_name),
            'YQL_TOKEN': self.get_vault_secret(
                self.Parameters.yql_token_vault_name),
            'STAT_TOKEN': self.get_vault_secret(
                self.Parameters.statface_token_vault_name)
        }
        os.environ.update(variables)

    def run_executable(self, resource_id, args):
        self.prepare_env()

        executable_path = self.get_resource_path(resource_id)

        if self.Parameters.yt_pool:
            args += [
                '--yt-pool', self.Parameters.yt_pool,
            ]

        args = [executable_path] + args

        with ProcessLog(self, 'Executable') as pl:
            return_code = subprocess.call(
                args, env=os.environ, stdout=pl.stdout, stderr=pl.stderr)

        if return_code != 0:
            raise TaskError(
                'Executable {} failed with {} return code'.format(
                    args, return_code))

    def on_execute(self):
        self.run_executable(
            self.Parameters.main_executable,
            self.Parameters.main_args)


class PoiMetricsCalculator(PoiMetricsExecutable):
    def find_resource(self, resource):
        if resource is None:
            raise ResourceNotFound('Resource {} not found'.format(resource))
        resource_data = sdk2.ResourceData(resource)
        return resource_data

    class Parameters(PoiMetricsExecutable.Parameters):
        with sdk2.parameters.Group('Executor executable') as executable_params:
            main_executable = sdk2.parameters.Resource(
                'Executor executable',
                resource_type=resources.PoiMetricsExecutorResource,
                required=False)
            executor_config = sdk2.parameters.Resource(
                'Json config',
                resource_type=resources.PoiMetricsExecutorConfigResource,
                required=False)
            processed_output_dir = sdk2.parameters.String(
                'Processed tables output dir',
                default='//home/maps/poi/statistics/processed/altay')
            main_args = None

        with sdk2.parameters.Group('Altay executables') as altay_metrics:
            bld_orgs_spread_exec = sdk2.parameters.Resource(
                'bld_orgs_spread executable resource id',
                resource_type=resources.PoiMetricsBldOrgsSpreadResource,
                required=False)
            glued_orgs_exec = sdk2.parameters.Resource(
                'glued_orgs executable resource id',
                resource_type=resources.PoiMetricsGluedOrgsResource,
                required=False)
            outside_buildings_orgs_exec = sdk2.parameters.Resource(
                'outside_buildings_orgs executable resource id',
                resource_type=resources.PoiMetricsOutsideBuildingsResource,
                required=False)
            reasons_exec = sdk2.parameters.Resource(
                'reasons executable resource id',
                resource_type=resources.PoiMetricsReasonsResource,
                required=False)
            org_conflicts_exec = sdk2.parameters.Resource(
                'org conflicts executable resource id',
                resource_type=resources.PoiMetricsOrgConflictsResource,
                required=False)
            org_movement_exec = sdk2.parameters.Resource(
                'org movement executable resource id',
                resource_type=resources.PoiMetricsOrgMovementResource,
                required=False)
            static_poi_export_exec = sdk2.parameters.Resource(
                'static_poi2nmaps executable resource id',
                resource_type=resources.PoiMetricsStaticPoiExportResource,
                required=False)
            output_dir = sdk2.parameters.String(
                'Output directory on Hahn',
                required=True,
                default='//home/maps/poi/statistics/altay')

        with sdk2.parameters.Group('Statface parameters') as statface_params:
            publish_report = sdk2.parameters.Bool('Publish report on Statface')

            with sdk2.parameters.RadioGroup('Statface proxy') as stat_proxy:
                stat_proxy.values.prod = stat_proxy.Value(
                    value='prod', default=True)
                stat_proxy.values.beta = stat_proxy.Value(
                    value='beta')

            report_title = sdk2.parameters.String('Report title')
            report_path = sdk2.parameters.String('Report path')

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_pool = sdk2.parameters.String(
                'YT procesing pool',
                default='maps-core-poi-statistics')

    def check_subtasks(self):
        if not all(
            task.status == ctt.Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def ensure_latest_resources_used(self):
        self.bld_orgs_spread_exec = self.Parameters.bld_orgs_spread_exec
        if not self.bld_orgs_spread_exec:
            self.bld_orgs_spread_exec = find_latest_resource(
                resources.PoiMetricsBldOrgsSpreadResource, self.Parameters.environment)

        self.glued_orgs_exec = self.Parameters.glued_orgs_exec
        if not self.glued_orgs_exec:
            self.glued_orgs_exec = find_latest_resource(
                resources.PoiMetricsGluedOrgsResource, self.Parameters.environment)

        self.outside_buildings_orgs_exec = self.Parameters.outside_buildings_orgs_exec
        if not self.outside_buildings_orgs_exec:
            self.outside_buildings_orgs_exec = find_latest_resource(
                resources.PoiMetricsOutsideBuildingsResource, self.Parameters.environment)

        self.reasons_exec = self.Parameters.reasons_exec
        if not self.reasons_exec:
            self.reasons_exec = find_latest_resource(
                resources.PoiMetricsReasonsResource, self.Parameters.environment)

        self.org_conflicts_exec = self.Parameters.org_conflicts_exec
        if not self.org_conflicts_exec:
            self.org_conflicts_exec = find_latest_resource(
                resources.PoiMetricsOrgConflictsResource, self.Parameters.environment)

        self.org_movement_exec = self.Parameters.org_movement_exec
        if not self.org_movement_exec:
            self.org_movement_exec = find_latest_resource(
                resources.PoiMetricsOrgMovementResource, self.Parameters.environment)

        self.static_poi_export_exec = self.Parameters.static_poi_export_exec
        if not self.static_poi_export_exec:
            self.static_poi_export_exec = find_latest_resource(
                resources.PoiMetricsStaticPoiExportResource, self.Parameters.environment)

        self.main_executable = self.Parameters.main_executable
        if not self.main_executable:
            self.main_executable = find_latest_resource(
                resources.PoiMetricsExecutorResource, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using PoiMetricsBldOrgsSpreadResource: %s',
            self.bld_orgs_spread_exec.id)
        logging.info(
            'Using PoiMetricsGluedOrgsResource: %s',
            self.glued_orgs_exec.id)
        logging.info(
            'Using PoiMetricsOutsideBuildingsResource: %s',
            self.outside_buildings_orgs_exec.id)
        logging.info(
            'Using PoiMetricsReasonsResource: %s',
            self.reasons_exec.id)
        logging.info(
            'Using PoiMetricsOrgConflictsResource: %s',
            self.org_conflicts_exec.id)
        logging.info(
            'Using PoiMetricsOrgMovementResource: %s',
            self.org_movement_exec.id)
        logging.info(
            'Using PoiMetricsStaticPoiExportResource: %s',
            self.static_poi_export_exec.id)
        logging.info(
            'Using PoiMetricsExecutorResource: %s',
            self.main_executable.id)

    def on_execute(self):
        self.ensure_latest_resources_used()

        with self.memoize_stage.metrics_stage:
            altay_executables_id = [
                self.bld_orgs_spread_exec,
                self.glued_orgs_exec,
                self.outside_buildings_orgs_exec,
                self.reasons_exec,
                self.org_conflicts_exec,
                self.org_movement_exec,
                self.static_poi_export_exec,
            ]

            altay_args = [
                '--output-dir',
                self.Parameters.output_dir
            ]

            logging.info('Launching child tasks')

            child_task_ids = []
            for executable in altay_executables_id:
                child_task = PoiMetricsExecutable(
                    self,
                    environment=self.Parameters.environment,
                    vault_owner=self.Parameters.vault_owner,
                    yt_token_vault_name=self.Parameters.yt_token_vault_name,
                    yql_token_vault_name=self.Parameters.yql_token_vault_name,
                    statface_token_vault_name=self.Parameters.statface_token_vault_name,
                    main_executable=executable,
                    main_args=altay_args,
                    description='Running executable {}'.format(executable),
                    juggler_report_is_enabled=self.Parameters.juggler_report_is_enabled,
                    juggler_host_name=self.Parameters.juggler_host_name,
                    juggler_service_name='child.' + self.Parameters.juggler_service_name,
                    yt_pool=self.Parameters.yt_pool)
                child_task_ids.append(child_task.enqueue())

            raise sdk2.WaitTask(
                child_task_ids,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True)

        self.check_subtasks()

        with self.memoize_stage.executor_stage:
            logging.info('All child tasks are finished')

            args = [
                '--input-folder', self.Parameters.output_dir,
                '--config-json', self.get_resource_path(
                    self.Parameters.executor_config),
                '--statface-proxy', self.Parameters.stat_proxy,
                '--report-title', self.Parameters.report_title,
                '--report-path', self.Parameters.report_path
            ]

            if self.Parameters.publish_report:
                args.append('--publish-report')

            processed_output_dir = self.Parameters.processed_output_dir
            if processed_output_dir:
                args += [
                    '--output-merged-table',
                    os.path.join(processed_output_dir, 'merged_table'),
                    '--save-folded-table',
                    os.path.join(processed_output_dir, 'folded_table'),
                ]

            self.run_executable(
                self.main_executable, args)

            logging.info('Statistics is published')
