import os

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqExecutableResourceWithAutoDiscovery


DATETIME_FORMAT = '%Y-%m-%d'


class PoiRanksCalculatorBinary(GeoqExecutableResourceWithAutoDiscovery):
    pass


class PoiRanksCalculator(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):
        with sdk2.parameters.Group('Executable') as executable_parameters:
            executable = sdk2.parameters.Resource(
                'Poi Ranks executable', resource_type=PoiRanksCalculatorBinary)

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_token = sdk2.parameters.YavSecret('YT OAuth token', required=True)
            experiments_token = sdk2.parameters.YavSecret('Experiments OAuth token', required=True)
            yt_pool = sdk2.parameters.String(
                'YT procesing pool',
                default='maps-core-poi-statistics')

        with sdk2.parameters.Group('Input') as input_parameters:
            ymapsdf_release = sdk2.parameters.String(
                'YMapsDF release folder on YT',
                default='//home/maps/core/garden/stable/ymapsdf/latest')
            extra_poi_bundle_release = sdk2.parameters.String(
                'Extra Poi Bundle release folder on YT',
                description='the latest will be used as a default')
            experiments_config_version = sdk2.parameters.Integer(
                'Experiments Maps Config `(config_id=43)` version to use',
                description='the latest will be used as a default',
                default=None)

        with sdk2.parameters.Group('Pipeline') as pipeline_parameters:
            poi_context_threshold = sdk2.parameters.Float(
                'Poi Context threshold',
                default=72,
                description=(
                    'Poi Neighbour search threshold in Mercator meters '
                    '(conflict distance for 16 zoom is used by default)'
                ))

        with sdk2.parameters.Group('Output') as output_parameters:
            output_date = sdk2.parameters.String(
                'Force output date',
                description='In {} Datetime format'.format(DATETIME_FORMAT))
            output_folder = sdk2.parameters.String('Output folder on', required=True)

    def on_execute(self):
        os.environ.update({
            'YT_TOKEN': self.Parameters.yt_token.value(),
            'EXPERIMENTS_TOKEN': self.Parameters.experiments_token.value()
        })

        date = self.Parameters.output_date or datetime.utcnow().strftime(DATETIME_FORMAT)
        output_path = os.path.join(self.Parameters.output_folder, date)

        arguments = [
            '--ymapsdf-release', self.Parameters.ymapsdf_release,
            '--output-table', output_path
        ]

        if self.Parameters.extra_poi_bundle_release:
            arguments += [
                '--extra-poi-bundle-release', self.Parameters.extra_poi_bundle_release
            ]

        if self.Parameters.experiments_config_version:
            arguments += [
                '--experiments-config-version', self.Parameters.experiments_config_version
            ]

        if self.Parameters.poi_context_threshold:
            arguments += [
                '--poi-context-threshold', self.Parameters.poi_context_threshold
            ]

        if self.Parameters.yt_pool:
            arguments += [
                '--pool', self.Parameters.yt_pool,
            ]

        self.run_executable(self.Parameters.executable, arguments)
