import os

from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.common.errors import ResourceNotFound, TaskError

from sandbox.sandboxsdk import environments
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)

DATETIME_FORMAT = '%Y-%m-%d'


class PoiProcessTilesBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PoiMergeZoomsBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PoiRendererImpressions(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Requirements):
        cores = 16
        ram = 8 * 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            tvm_secret_vault = sdk2.parameters.Vault('TVM secret vault', required=True)
            yt_pool = sdk2.parameters.String(
                'YT processing pool',
                default='maps-core-poi-statistics',
                required=True)

        with sdk2.parameters.Group('Binaries'):
            process_tiles_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/impressions/renderer/process_tiles binary',
                resource_type=PoiProcessTilesBinary,
                required=True)

            merge_zooms_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/impressions/renderer/merge_zooms binary',
                resource_type=PoiMergeZoomsBinary,
                required=True)

        with sdk2.parameters.Group('Input parameters'):
            positions_table = sdk2.parameters.String(
                'Positions table on Hahn',
                required=True)

            min_zoom = sdk2.parameters.Integer(
                'Min zoom',
                default=2)

            max_zoom = sdk2.parameters.Integer(
                'Max zoom',
                default=21)

            lon = sdk2.parameters.String(
                'Column name for lon',
                default='lon')

            lat = sdk2.parameters.String(
                'Column name for lat',
                default='lat')


        with sdk2.parameters.Group('Output parameters'):
            output_table = sdk2.parameters.String(
                'Output path on Hahn',
                required=True)

        with sdk2.parameters.Group('Process tiles parameters'):
            zooms_table = sdk2.parameters.String(
                'Output unpacked-zooms table on Hahn (temporary table will be used if not specified)',
                default='')

            tile_params = sdk2.parameters.String(
                'Output tile-params table on Hahn (temporary table will be used if not specified)',
                default='')

            renderer_url = sdk2.parameters.String(
                'Renderer url',
                default='http://core-renderer-tiles.maps.yandex.net',
                required=True)

            threads_number = sdk2.parameters.Integer(
                'Number of threads',
                default=16)

            self_tvm_id = sdk2.parameters.Integer(
                'Current TVM app id',
                default=2025240,
                required=True)

            renderer_tvm_id = sdk2.parameters.Integer(
                'Renderer TVM app id',
                default=2023928,
                required=True)

            max_rps = sdk2.parameters.Integer(
                'Max RPS',
                default=2000)

        with sdk2.parameters.Group('Merge zooms parameters'):
            date = sdk2.parameters.String(
                'Current date to insert into table '
                '(current date in {} format will be used if not specified)'.format(DATETIME_FORMAT),
                default='')

    def run_executable(self, executable_resource, executable_arguments):
        if not executable_resource:
            raise ResourceNotFound('Executable resource was not found. Please provide a valid executable.')

        # ensure every argument is str
        executable_arguments = [str(arg) for arg in executable_arguments]

        executable = str(sdk2.ResourceData(executable_resource).path)
        args = [executable] + executable_arguments

        with ProcessLog(self, 'Executable') as pl:
            return_code = subprocess.call(
                args, stdout=pl.stdout, stderr=pl.stderr)

        if return_code != 0:
            raise TaskError(
                'Executable {} failed with {} return code'.format(
                    args, return_code))

    def on_execute(self):
        current_date = datetime.utcnow().strftime(DATETIME_FORMAT)

        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        os.environ['YT_PROXY'] = 'hahn'
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['IMPRESSIONS_TVM_SECRET'] = self.Parameters.tvm_secret_vault.data()

        unpacked_zooms_table = self.Parameters.zooms_table
        if not unpacked_zooms_table:
            unpacked_zooms_table = yt.create_temp_table()

        tile_params_table = self.Parameters.tile_params
        if not tile_params_table:
            tile_params_table = yt.create_temp_table()

        date = self.Parameters.date or current_date
        output_table = yt.ypath_join(self.Parameters.output_table, date)

        # running process_tiles
        process_tiles_args = [
            # output table
            '--destination', unpacked_zooms_table,
            # tile params table
            '--tile-params', tile_params_table,
            # input table
            '--positions', self.Parameters.positions_table,
            # renderer url
            '--source', self.Parameters.renderer_url,
            # zooms interval to fetch
            '--min-zoom', self.Parameters.min_zoom,
            '--max-zoom', self.Parameters.max_zoom,
            # number of threads
            '--threads-num', self.Parameters.threads_number,
            # tvm parameters (using default tvm cache directory)
            '--tvm-self-id', self.Parameters.self_tvm_id,
            '--tvm-renderer-id', self.Parameters.renderer_tvm_id,
            # rps
            '--max-rps', self.Parameters.max_rps,
            # lon&lat column names
            '--lon', self.Parameters.lon,
            '--lat', self.Parameters.lat,
        ]

        self.run_executable(self.Parameters.process_tiles_binary, process_tiles_args)

        # running merge zooms
        merge_zooms_args = [
            # output table
            '--impressions', output_table,
            # output table from the previous step
            '--unpacked-zooms', unpacked_zooms_table,
            # tile params table from the previous step
            '--tile-params', tile_params_table,
            # positions table from the previous step
            '--poi-table', self.Parameters.positions_table,
            # zooms interval to fetch
            '--min-zoom', self.Parameters.min_zoom,
            '--max-zoom', self.Parameters.max_zoom,
            # lon&lat column names
            '--lon', self.Parameters.lon,
            '--lat', self.Parameters.lat,
            # date
            '--date', date,
            # yt processing pool
            '--yt-pool', self.Parameters.yt_pool,
        ]

        self.run_executable(self.Parameters.merge_zooms_binary, merge_zooms_args)
