import os

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqExecutableResourceWithAutoDiscovery


DATETIME_FORMAT = '%Y-%m-%d'


class GeoqRepresentativeMetricsExecutable(GeoqExecutableResourceWithAutoDiscovery):
    pass


class GeoqRepresentativeMetrics(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):
        with sdk2.parameters.Group('Executable') as executable_parameters:
            executable = sdk2.parameters.Resource(
                'Metrics pipeline executable', resource_type=GeoqRepresentativeMetricsExecutable)

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_token = sdk2.parameters.YavSecret('YT OAuth token', required=True)
            stat_token = sdk2.parameters.YavSecret('Stat OAuth token', required=True)
            yt_pool = sdk2.parameters.String(
                'YT procesing pool',
                default='maps-core-poi-statistics')

        with sdk2.parameters.Group('Input') as input_parameters:
            altay_company_export = sdk2.parameters.String(
                'Altay Company Export table',
                default='//home/altay/db/export/current-state/snapshot/company')
            orginfo = sdk2.parameters.String(
                'Orginfo table',
                default='//home/maps/poi/data/orginfo')
            org_basemap_reasons = sdk2.parameters.String(
                'Organizations reasons not to be on basemap',
                default='//home/maps/poi/data/org_basemap_reasons')
            ymapsdf_release = sdk2.parameters.String(
                'YaMapsDF release folder',
                default='//home/maps/core/garden/stable/ymapsdf/latest')
            extra_poi_bundle = sdk2.parameters.String(
                'Extra Poi Bundle table',
                description='leave it empty to use the most recent one')
            renderer_impressions = sdk2.parameters.String(
                'Renderer Impressions table',
                description='leave it empty to use the most recent one')

        with sdk2.parameters.Group('Statface') as statface_parameters:
            statface_report_date = sdk2.parameters.String(
                'Force report date (in format {})'.format(DATETIME_FORMAT))
            statface_report_path = sdk2.parameters.String(
                'Statface report path',
                required=True)

            with sdk2.parameters.RadioGroup('Proxy') as statface_proxy:
                statface_proxy.values.prod = statface_proxy.Value(value='prod')
                statface_proxy.values.beta = statface_proxy.Value(value='beta', default=True)

    def on_execute(self):
        os.environ.update({
            'YT_TOKEN': self.Parameters.yt_token.value(),
            'STAT_TOKEN': self.Parameters.stat_token.value()
        })

        date = self.Parameters.statface_report_date or datetime.utcnow().strftime(DATETIME_FORMAT)
        arguments = [
            '--altay-company-export-table', self.Parameters.altay_company_export,
            '--orginfo-table', self.Parameters.orginfo,
            '--org-basemap-reasons-table', self.Parameters.org_basemap_reasons,
            '--ymapsdf-release', self.Parameters.ymapsdf_release,
            '--statface-report-date', date,
            '--statface-report-path', self.Parameters.statface_report_path,
            '--statface-proxy', self.Parameters.statface_proxy
        ]

        if self.Parameters.extra_poi_bundle:
            arguments += ['--extra-poi-bundle-table', self.Parameters.extra_poi_bundle]

        if self.Parameters.renderer_impressions:
            arguments += ['--renderer-impressions-table', self.Parameters.renderer_impressions]

        if self.Parameters.yt_pool:
            arguments += [
                '--yt-pool', self.Parameters.yt_pool,
            ]

        self.run_executable(self.Parameters.executable, arguments)
