import os

from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.common.errors import ResourceNotFound, TaskError
from sandbox.common.types import task as ctt

from sandbox.sandboxsdk import environments

from sandbox.projects.maps.poi.PoiRendererImpressions import (
    PoiProcessTilesBinary,
    PoiMergeZoomsBinary,
)

from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)


DATETIME_FORMAT = '%Y-%m-%d'


class YellowZonesMetricsBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class YellowZonesCollectCoordinatesBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class YellowZonesMetrics(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.Group('Binaries') as binaries:
            metrics_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/yellow_zones/statistics binary',
                resource_type=YellowZonesMetricsBinary,
                required=True)

            collect_coordinates_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/yellow_zones/collect_coordinates binary',
                resource_type=YellowZonesCollectCoordinatesBinary,
                required=True)

            process_tiles_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/impressions/renderer/process_tiles binary',
                resource_type=PoiProcessTilesBinary,
                required=True)

            merge_zooms_binary = sdk2.parameters.Resource(
                'maps/poi/statistics/impressions/renderer/merge_zooms binary',
                resource_type=PoiMergeZoomsBinary,
                required=True)

        with sdk2.parameters.Group('Tokens') as tokens:
            yt_token_vault = sdk2.parameters.Vault(
                'YT token name',
                required=True)

            statface_token_vault = sdk2.parameters.Vault(
                'Statface token name',
                required=True)

            tvm_secret_vault = sdk2.parameters.Vault(
                'TVM secret vault',
                required=True)

        with sdk2.parameters.Group('Input tables') as input_tables:
            orginfo_table = sdk2.parameters.String(
                'Orginfo table',
                default='//home/maps/poi/data/orginfo',
                required=True)

            org_statistics_table = sdk2.parameters.String(
                'Org statistics table',
                default='//home/maps/poi/data/org_statistics',
                required=True)

        with sdk2.parameters.Group('Output tables') as output_tables:
            output_folded_table = sdk2.parameters.String(
                'Output folded table on Hahn',
                required=True)

            output_renderer_impressions_folder = sdk2.parameters.String(
                'Output impressions folder on Hahn',
                required=True)

            output_renderer_impressions_force_date = sdk2.parameters.String(
                'Date to use in {} format (current date will be used if not specified)'.format(DATETIME_FORMAT),
                default='')

        with sdk2.parameters.Group('Statface parameters') as statface_params:
            with sdk2.parameters.RadioGroup('Proxy') as proxy:
                proxy.values.prod = proxy.Value(value='stat')
                proxy.values.beta = proxy.Value(value='stat-beta', default=True)

            report_path = sdk2.parameters.String(
                'Report path (leave empty if you don\'t want to send it)')

        with sdk2.parameters.Group('Renderer impressions parameters') as collect_coordinates_parameters:
            coordinates = sdk2.parameters.String(
                'Output collected coordinates table on Hahn (temporary table will be used if not specified)',
                default='')

            min_zoom = sdk2.parameters.Integer('Min zoom', default=10)
            max_zoom = sdk2.parameters.Integer('Max zoom', default=21)

    def check_subtasks(self):
        if not all(
            task.status == ctt.Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def run_executable(self, executable_resource, executable_arguments):
        if not executable_resource:
            raise ResourceNotFound('Executable resource was not found. Please provide a valid executable.')

        # ensure every argument is str
        executable_arguments = [str(arg) for arg in executable_arguments]

        executable = str(sdk2.ResourceData(executable_resource).path)
        args = [executable] + executable_arguments

        with ProcessLog(self, 'Executable') as pl:
            return_code = subprocess.call(
                args, stdout=pl.stdout, stderr=pl.stderr)

        if return_code != 0:
            raise TaskError(
                'Executable {} failed with {} return code'.format(
                    args, return_code))

    def on_execute(self):
        with self.memoize_stage.prepare:
            current_date = datetime.utcnow().strftime(DATETIME_FORMAT)
            self.Context.date = self.Parameters.output_renderer_impressions_force_date or current_date

        yt_token = self.Parameters.yt_token_vault.data()
        variables = {
            'YT_TOKEN': yt_token,
            'STAT_TOKEN': self.Parameters.statface_token_vault.data(),
            'YT_PROXY': 'hahn'
        }
        os.environ.update(variables)

        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = yt_token

        self.check_subtasks()

        renderer_impressions_table = yt.ypath_join(
            self.Parameters.output_renderer_impressions_folder, self.Context.date)

        with self.memoize_stage.yellow_zones_statistics:
            metrics_args = [
                # maps/poi/data tables
                '--orginfo-table', self.Parameters.orginfo_table,
                '--org-statistics-table', self.Parameters.org_statistics_table,
                # very helpful debug output
                '--output-folded-table', self.Parameters.output_folded_table,
                # statface proxy to use (prod or beta)
                '--statface-proxy', self.Parameters.proxy,
                # renderer impressions table from the previous stage
                '--renderer-impressions-table', renderer_impressions_table,
                # zooms
                '--min-zoom', self.Parameters.min_zoom,
                '--max-zoom', self.Parameters.max_zoom
            ]

            if self.Parameters.report_path:
                metrics_args += ['--statface-report-path', self.Parameters.report_path]

            self.run_executable(self.Parameters.metrics_binary, metrics_args)
