# -*- coding: utf-8 -*-

import json
import logging
import tempfile
import pipes

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp


class QtoolsDeployApplication(sdk2.Task):
    """ Task for deploying of application to qloud using of qtools """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 1 * 1024
        disk_space = 2 * 1024

        environments = [environments.NodeJS("10.14.2")]

    class Parameters(sdk2.Task.Parameters):
        config_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url to qtools config file", required=True,
        )

        config_revision = sdk2.parameters.String(
            "Revision of config", default_value=None
        )

        registry_tag = sdk2.parameters.String(
            "Tag of docker image", default_value=None
        )

        with sdk2.parameters.RadioGroup(
            "Qloud environment of application", required=True
        ) as qloud_environment:
            qloud_environment.values.testing = qloud_environment.Value(
                default=True
            )
            qloud_environment.values.production = None

        with sdk2.parameters.RadioGroup(
            "Deploy target state", required=True
        ) as qloud_target_state:
            qloud_target_state.values.DEPLOYED = qloud_target_state.Value(
                default=True
            )
            qloud_target_state.values.PREPARED = None
            qloud_target_state.values.COMMITTED = None

        vault_item_owner = sdk2.parameters.String(
            "Qtools token vault secret item owner", required=True
        )

        vault_item_name = sdk2.parameters.String(
            "Qtools token vault secret item name", required=True
        )

        qloud_deploy_comment = sdk2.parameters.String(
            "Qloud deploy comment", required=False
        )

    def on_execute(self):
        tmpdir = tempfile.mkdtemp()

        self._prepare_package_json(tmpdir)
        self._prepare_qtools_config(tmpdir)
        self._install_requirements(tmpdir)
        self._run_deploy(tmpdir)

    @property
    def _node_environment(self):
        return self.Requirements.environments[0]

    @property
    def _node_path(self):
        return self._node_environment.node

    @property
    def _npm_path(self):
        return self._node_environment.npm

    @property
    def _qtools_token(self):
        return sdk2.Vault.data(
            self.Parameters.vault_item_owner,
            self.Parameters.vault_item_name,
        )

    @property
    def _config_revision(self):
        if self.Parameters.config_revision:
            return self.Parameters.config_revision

        return None

    @property
    def _registry_tag(self):
        if self.Parameters.registry_tag:
            return self.Parameters.registry_tag
        return None

    def _prepare_qtools_config(self, workspace):
        qtools_config = json.loads(
            sdk2.svn.Arcadia.cat(
                self.Parameters.config_url, revision=self._config_revision
            )
        )

        if self._registry_tag:
            qtools_config["registry"]["tag"] = self._registry_tag

        qtools_config_path = sdk2.path.Path(workspace, ".qtools.json")
        with open(str(qtools_config_path), "w") as out:
            config_data = json.dumps(qtools_config, indent=4)

            logging.info(".qtools.json: {} ".format(config_data))
            out.write(config_data)

        return str(qtools_config_path)

    def _prepare_package_json(self, workspace):
        package_json = {
            "scripts": {},
            "devDependencies": {"@yandex-int/qtools": "^0.11.0"},
            "dependencies": {},
        }

        package_path = sdk2.path.Path(workspace, "package.json")
        with open(str(package_path), "w") as out:
            package_data = json.dumps(package_json, indent=4)

            logging.info("package.json: {}".format(package_data))
            out.write(package_data)

        return str(package_path)

    def _install_requirements(self, cwd):
        self._run_npm(
            ["install", "--registry=https://npm.yandex-team.ru"], cwd=cwd
        )

    def _run_deploy(self, cwd):
        cmd = [
            self._node_path,
            "node_modules/.bin/qtools",
            "deploy",
            self.Parameters.qloud_environment,
            "--force",
            "--wait",
            "-v2",
            "--target-state",
            self.Parameters.qloud_target_state,
        ]

        if self.Parameters.qloud_deploy_comment:
            cmd += ["--comment", pipes.quote(self.Parameters.qloud_deploy_comment)]

        logger = logging.getLogger("node.qtools")
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            sp.check_call(
                " ".join(map(str, cmd)),
                env=dict(QTOOLS_TOKEN=self._qtools_token, INIT_CWD=cwd),
                cwd=cwd,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

    def _run_npm(self, cmd, cwd, env=None):
        logger = logging.getLogger("npm.{}".format(cmd[0]))

        cmd = [self._node_path, self._npm_path] + cmd

        npm_env = {}
        npm_env.update(env if env is not None else {})

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            sp.check_call(
                " ".join(map(str, cmd)),
                env=npm_env,
                cwd=cwd,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
