from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sandboxsdk.process import run_process

import logging
import os
import subprocess


def _builds_stubs(arc_root, path):
    run_process(
        [
            os.path.join(arc_root, "ya"),
            "make",
            os.path.join(arc_root, path),
            "--add-result=.pyi",
            "--replace-result",
        ],
        log_prefix="build_proto_library",
        outputs_to_one_file=True,
    )


def _get_proto_libraries(arc_root, module):
    ya_dump = run_process(
        [
            os.path.join(arc_root, "ya"),
            "dump",
            "modules",
            "--no-tools",
            os.path.join(arc_root, module.replace(".", "/")),
        ],
        stdout=subprocess.PIPE,
    )
    # 'ya dump module' output example:
    # module: Library maps/libs/common $B/maps/libs/common/libmaps-libs-common.a
    # module: Library maps/garden/sdk/module_rpc/proto $B/maps/garden/sdk/module_rpc/proto/libpy3sdk-module_rpc-proto.a
    library_prefix = "module: Library "
    return {
        x.removeprefix(library_prefix).split()[0]
        for x in ya_dump.communicate()[0].decode("utf-8").splitlines()
        if x.startswith(library_prefix) and "proto" in x
    }


def _generate_pyis(arc_root, module):
    files_to_patch = set()
    for lib in _get_proto_libraries(arc_root, module):
        _builds_stubs(arc_root, lib)
        full_path = os.path.join(arc_root, lib)
        for subdir, _, files in os.walk(full_path):
            files_to_patch.update(
                [
                    os.path.join(full_path, subdir, file)
                    for file in files
                    if file.endswith(".pyi")
                ]
            )
    # '\x' in comments leads to error in mypy check, need to replace it
    for file in files_to_patch:
        with open(file, "r+") as f:
            buffer = f.read()
            f.seek(0)
            f.truncate()
            f.write(buffer.replace(r"\x", r"\\x"))


class BizdirMypyTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        privileged = True
        container_resource = 2649223535

    class Parameters(sdk2.Task.Parameters):
        module_name = sdk2.parameters.String(
            "Module name for mypy check",
            default="maps.bizdir.sps",
        )
        config_file = sdk2.parameters.String(
            "Path to config file in Arcadia",
            default="maps/bizdir/sps/pyproject.toml",
        )
        commit_hash = sdk2.parameters.String("Arc commit hash", required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @sdk2.header(title="Mypy check report")
    def show_report(self):
        COLOR_GREEN = "#008000"
        COLOR_RED = "#FF0000"
        COLOR_BLUE = "#42AAFF"

        def badge(text, color):
            return (
                '<span style="background-color: %s; color: #ffffff; font-weight: bold; padding: 0px 8px">%s</span>'
                % (color, text)
            )

        if self.status not in list(Status.Group.FINISH + Status.Group.BREAK):
            return badge("Mypy check not finished", COLOR_BLUE)

        result_badge = (
            badge("Mypy FAILED", COLOR_RED)
            if self.Context.mypy_failed
            else badge("Mypy OK", COLOR_GREEN)
        )

        return str(
            result_badge + "\n\nMypy output:\n\n" + self.Context.mypy_output
        ).replace("\n", "<br>")

    def run_mypy(self, arc_root):
        env = os.environ.copy()
        env["ARCADIA_ROOT"] = arc_root
        try:
            with sdk2.helpers.ProcessLog(self, logger="mypy") as fout:
                log_file = str(fout.stdout.path)
                run_process(
                    [
                        "python3.9",
                        "-m",
                        "mypy",
                        "--config-file",
                        self.Parameters.config_file,
                        "-p",
                        self.Parameters.module_name,
                    ],
                    work_dir=arc_root,
                    environment=env,
                    outputs_to_one_file=True,
                    stdout=fout.stdout,
                )
        except Exception:
            logging.exception("Mypy check failed")
            raise TaskFailure("Mypy check failed")
        else:
            self.Context.mypy_failed = False
        finally:
            with open(log_file, "r") as mypy_out:
                self.Context.mypy_output = mypy_out.read()

    def on_execute(self):
        super(BizdirMypyTest, self).on_execute()

        logging.info("Starting...")

        self.Context.mypy_failed = True
        with arcadia_sdk.mount_arc_path(
            "arcadia-arc://#" + self.Parameters.commit_hash
        ) as arc_root:
            logging.info("Generating proto pyis...")
            _generate_pyis(arc_root, self.Parameters.module_name)
            logging.info("Running mypy...")
            self.run_mypy(arc_root)
        logging.info("Finished")
