from logging import getLogger, info

from datetime import datetime, timedelta
from pytz import utc
from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.projects import resource_types
from sandbox.projects.common import task_env
from sandbox.projects.mapsearch.common.find_resource import find_latest_resource
from sandbox.projects.mapsearch.common.juggler import juggler_ok
from sandbox.projects.tycoon.TycoonAdverts import TycoonAds
from sandbox.sdk2.helpers import subprocess as sp


class BuildMapsDatabaseAdvert1(sdk2.Task):
    '''Produces menu_advert.pb.bin and advert_tags.pb.bin for basesearch'''

    class Requirements(task_env.TinyRequirements):
        disk_space = 256
        ram = 1024 * 3

    def on_execute(self):
        info('BuildMapsDatabaseAdvert1 launched')

        indexer = find_latest_resource(resource_types.GEOADVERTINDEXER_EXECUTABLE, {'released': 'stable'})[1]

        last_index = sdk2.Resource.find(resource_types.MAPS_DATABASE_ADVERT, state=[State.READY], attrs={'released': 'stable'}).first()
        last_tycoon_id = int(getattr(last_index, 'last_tycoon_id', -1))

        info('Last processed tycoon ID: ' + str(last_tycoon_id))

        info('Collecting tycoons...')

        tycoons = []
        expires = datetime.now(utc) - timedelta(hours=1)

        for tycoon in sdk2.Resource.find(TycoonAds, state=[State.READY], attrs={'released': 'stable'}).limit(40):
            tycoons.append(tycoon)
            if tycoon.created < expires:
                break

        info('Collected tycoon IDs: ' + ', '.join(str(tycoon.id) for tycoon in tycoons))

        if len(tycoons) == 0 or tycoons[0].id <= last_tycoon_id:
            self.set_info('No new tycoons. Nothing to do')
            juggler_ok('menu_tags_indexing_alive', 'Indexing skipped (nothing to do)')
            return

        info('Filtering tycoons with equal advert tags...')

        unique_tycoons = []
        advert_tag_hashes = set()

        for tycoon in tycoons:
            if tycoon.advert_tag_hash not in advert_tag_hashes:
                unique_tycoons.append(tycoon)
                advert_tag_hashes.add(tycoon.advert_tag_hash)

        info('Tycoon IDs to be processed: ' + ', '.join(str(tycoon.id) for tycoon in unique_tycoons))

        info('Launching indexer...')

        index_dir = 'index'
        filenames = [str(sdk2.ResourceData(tycoon).path.joinpath('ads.xml')) for tycoon in unique_tycoons]
        command = ' '.join([indexer, '-d', index_dir] + filenames)

        with sdk2.helpers.ProcessLog(self, logger=getLogger('indexing_logger')) as pl:
            sp.Popen(command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()

        index = resource_types.MAPS_DATABASE_ADVERT(self, self.Parameters.description, index_dir)
        index.last_tycoon_id = unique_tycoons[0].id
        index.released = 'stable'
        index.ttl = 60
        sdk2.ResourceData(index).ready()

        info('Indexing completed')
        self.set_info('Tycoons indexed: ' + str(len(unique_tycoons)))

        juggler_ok('menu_tags_indexing_alive', 'Indexing completed')
        juggler_ok('menu_tags_freshness', 'Index updated')
        info('BuildMapsDatabaseAdvert1 finished')
