import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments


class UpdateAdvertisedOrgsPoints(sdk2.Task):
    '''Updates location and additional data (name, rubric) for advertised geoproduct orgs'''

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yql')
        ]

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        logging.info('UpdateAdvertisedOrgsPoints launched')

        logging.info('Creating YQL client...')
        yql_client = YqlClient(db='hahn', token=sdk2.Vault.data('robot-maps-search', 'YQL_TOKEN'))
        logging.info('YQL client created')

        logging.info('Calculating advertised orgs points...')
        query_text = '''
$max_pin_limit = 1000;

$get_rubric_ids_list = Python::get_rubric_ids_list(Callable<(List<Struct<'Id':Uint64,'IsMain':Bool?,'SnippetType':String?>>?)->List<Uint64>>, @@
def get_rubric_ids_list(rubric_list):
    return [r.Id for r in rubric_list]
@@);

$rubric_class_reducer = Python::rubric_class_reducer(Callable<(Int64?,Stream<Utf8?>)->Struct<permalink:Int64,rubric_class:Utf8>>, @@
class Row(object):
    def __init__(self, permalink, rubric_class):
        self.permalink = permalink
        self.rubric_class = rubric_class

def rubric_class_reducer(permalink, rubric_classes):
    for rubric_class in rubric_classes:
        if rubric_class is not None:
            return Row(permalink, rubric_class)
    return Row(permalink, '')
@@);

$advertised_orgs_data = (
    SELECT
        company.permalink AS permalink,
        CAST(company.exported_company.Name[0].Value AS Utf8) AS name,
        CAST(company.exported_company.ShortName[0].Value AS Utf8) AS short_name,
        company.exported_company.Geo.Location.Pos.Lon AS lon,
        company.exported_company.Geo.Location.Pos.Lat AS lat,
        company.exported_company.Rubric AS rubric
    FROM
        `home/maps-search/toponym_adverts/toponym_adverts_data/advertised_orgs_ids` AS advertised_orgs_ids
    INNER JOIN
        (select * from `home/altay/db/export/current-state/exported/company` where not Coalesce(exported_company.IsOnline, False)) AS company
    ON
        company.permalink = advertised_orgs_ids.id
);

$advertised_orgs_names_with_nums = (
    SELECT
        org.name AS name,
        COUNT(*) AS num
    FROM
        $advertised_orgs_data AS org
    GROUP BY
        org.name
);

$showable_orgs_data_1 = (
    SELECT
        org.permalink AS permalink,
        org.name AS name,
        org.short_name AS short_name,
        org.lon AS lon,
        org.lat AS lat,
        org.rubric AS rubric,
        $get_rubric_ids_list(org.rubric) AS rubric_id
    FROM
        $advertised_orgs_data AS org
    JOIN
        $advertised_orgs_names_with_nums AS nums
    ON
        org.name = nums.name
    WHERE
        nums.num <= $max_pin_limit
);

$bad_rubrics = (
    SELECT
        ToSet(AGGREGATE_LIST(id))
    FROM
        `home/maps-search/toponym_adverts/toponym_adverts_data/bad_rubrics`
);

$showable_orgs_data_2 = (
    SELECT
        org.permalink AS permalink,
        org.name AS name,
        org.short_name AS short_name,
        org.lon AS lon,
        org.lat AS lat,
        org.rubric AS rubric,
        org.rubric_id AS rubric_id
    FROM
        $showable_orgs_data_1 AS org
    WHERE
        SetIsDisjoint($bad_rubrics, ToSet(rubric_id))
);

$flat_permalinks_with_rubric_ids = (
    SELECT
        org.permalink AS permalink,
        org.rubric_id AS rubric_id
    FROM
        $showable_orgs_data_2 AS org
    FLATTEN BY
        rubric_id
);

$flat_permalinks_with_rubric_classes = (
    SELECT
        pr.permalink AS permalink,
        icon.rubric_class AS rubric_class
    FROM
        $flat_permalinks_with_rubric_ids AS pr
    LEFT JOIN
        `home/altay/db/export/current-state/exported/rubric` AS rubric
    ON
        pr.rubric_id = rubric.exported_rubric.Id
    LEFT JOIN
        `home/maps-search/toponym_adverts/toponym_adverts_data/rubric_classes_with_icons` AS icon
    ON
        rubric.exported_rubric.RubricClass = icon.rubric_class
);

$permalinks_with_rubric_classes = (
    REDUCE
        $flat_permalinks_with_rubric_classes
    ON
        permalink
    USING
        $rubric_class_reducer(rubric_class)
);

INSERT INTO
    `home/maps-search/toponym_adverts/toponym_adverts_data/advertised_orgs_points`
WITH TRUNCATE
SELECT
    org.permalink AS permalink,
    org.name AS name,
    org.short_name AS short_name,
    org.lon AS lon,
    org.lat AS lat,
    pr.rubric_class AS rubric_class
FROM
    $showable_orgs_data_2 AS org
LEFT JOIN
    $permalinks_with_rubric_classes AS pr
ON
    org.permalink = pr.permalink
;
        '''
        request = yql_client.query(query_text, syntax_version=1)
        request.run()
        request.get_results()
        logging.info('Advertised orgs points calculation complete')

        logging.info('UpdateAdvertisedOrgsPoints finished')
