import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments


class UpdateToponymBoundingBoxes(sdk2.Task):
    '''Finds actual YMapsDF/cis1 source in YT and calculates bounding boxes for houses, streets, districts and localities'''

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yql'),
        ]

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        logging.info('UpdateToponymBoundingBoxes launched')

        logging.info('Creating YQL client...')
        yql_client = YqlClient(db='hahn', token=sdk2.Vault.data('robot-maps-search', 'YQL_TOKEN'))
        logging.info('YQL client created')

        logging.info('Calculating toponym bounding boxes...')
        query_text = '''
$ymapsdf         = "home/maps/core/garden/stable/ymapsdf/latest";
$ymapsdf_ad      = $ymapsdf || "/cis1/ad";
$ymapsdf_ad_geom = $ymapsdf || "/cis1/ad_geom";
$ymapsdf_addr    = $ymapsdf || "/cis1/addr";
$ymapsdf_node    = $ymapsdf || "/cis1/node";
$ymapsdf_rd      = $ymapsdf || "/cis1/rd";
$ymapsdf_rd_geom = $ymapsdf || "/cis1/rd_geom";

$get_bounding_box = Python::get_bounding_box(Callable<(String?)->Tuple<Float,Float,Float,Float>>, @@
import codecs
import shapely.wkb

SPAN_WIDTH, SPAN_HEIGHT = 0.015, 0.005

def get_bounding_box(wkb_encoded_geometry):
    geometry = shapely.wkb.loads(codecs.decode(wkb_encoded_geometry, 'hex'))
    min_lon, min_lat, max_lon, max_lat = None, None, None, None
    if geometry.type == 'Point':
        min_lon, min_lat = geometry.x - SPAN_WIDTH / 2, geometry.y - SPAN_HEIGHT / 2
        max_lon, max_lat = geometry.x + SPAN_WIDTH / 2, geometry.y + SPAN_HEIGHT / 2
    else:
        points = []
        if geometry.type == 'Polygon':
            points += geometry.exterior.coords
        elif geometry.type == 'MultiPolygon':
            for geometry_part in geometry:
                points += geometry_part.exterior.coords
        elif geometry.type == 'LineString':
            points += geometry.coords
        elif geometry.type == 'MultiLineString':
            for geometry_part in geometry:
                points += geometry_part.coords
        for point in points:
            if min_lon is None or point[0] < min_lon:
                min_lon = point[0]
            if min_lat is None or point[1] < min_lat:
                min_lat = point[1]
            if max_lon is None or point[0] > max_lon:
                max_lon = point[0]
            if max_lat is None or point[1] > max_lat:
                max_lat = point[1]
    return min_lon, min_lat, max_lon, max_lat
@@);

$houses_data = (
    SELECT
        addr.addr_id AS geocoder_id, 'house' AS kind, $get_bounding_box(node.shape) AS bounding_box
    FROM
        $ymapsdf_addr AS addr JOIN $ymapsdf_node AS node ON addr.node_id = node.node_id
);

$streets_data = (
    SELECT
        rd.rd_id AS geocoder_id, 'street' AS kind, $get_bounding_box(rd_geom.shape) AS bounding_box
    FROM
        $ymapsdf_rd AS rd JOIN $ymapsdf_rd_geom AS rd_geom ON rd.rd_id = rd_geom.rd_id
    WHERE
        rd.rd_type = 1
);

$localities_and_districts_data = (
    SELECT
        ad.ad_id AS geocoder_id, (CASE WHEN ad.level_kind = 4 THEN 'locality' ELSE 'district' END) AS kind, $get_bounding_box(ad_geom.shape) AS bounding_box
    FROM
        $ymapsdf_ad AS ad JOIN $ymapsdf_ad_geom AS ad_geom ON ad.ad_id = ad_geom.ad_id
    WHERE
        ad.level_kind = 4 OR ad.level_kind = 5
);

$toponyms_data = (
    (SELECT * FROM $houses_data)
    UNION ALL
    (SELECT * FROM $streets_data)
    UNION ALL
    (SELECT * FROM $localities_and_districts_data)
);

INSERT INTO
    `home/maps-search/toponym_adverts/toponym_adverts_data/toponym_bounding_boxes`
WITH TRUNCATE
SELECT
    geocoder_id,
    CAST(kind AS Utf8) AS kind,
    bounding_box.0 AS min_lon,
    bounding_box.1 AS min_lat,
    bounding_box.2 AS max_lon,
    bounding_box.3 AS max_lat
FROM
    $toponyms_data
;
        '''
        request = yql_client.query(query_text, syntax_version=1)
        request.run()
        request.get_results()
        logging.info('Toponym bounding boxes calculation complete')

        logging.info('UpdateToponymBoundingBoxes finished')
