# coding=utf-8
import logging

from sandbox import sdk2
from sandbox.sdk2 import environments


class CheckOdPdMonitoring(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("WMS database settings", collapse=True) as wms_db_settings:
            database_user_secret = sdk2.parameters.YavSecret(
                'WMS database username'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: username',
                required=True
            )
            database_password_secret = sdk2.parameters.YavSecret(
                'WMS database password '
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: password',
                required=True
            )
            database_host_string = sdk2.parameters.String('host1, host2,...', required=True)
            database_port_string = sdk2.parameters.String('port1, port2,...', required=True)
            database_warehouse_string = sdk2.parameters.String('warehouse1, warehouse2,...', required=True)

        with sdk2.parameters.Group("Startrek settings", collapse=True) as st_settings:
            st_user_agent = sdk2.parameters.String('Startrek user agent', required=True)
            st_url = sdk2.parameters.String('Startrek url', required=True)
            st_token_secret = sdk2.parameters.YavSecret(
                'Startrek token'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: token',
                required=True
            )
            st_queue = sdk2.parameters.String('Startrek queue', required=True)
            st_summary = sdk2.parameters.String('Startrek ticket summary', required=True)
            st_tags = sdk2.parameters.List('List of tags for Startrek issue')
            board = sdk2.parameters.String('Board URL')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            environments.PipEnvironment('pymssql==2.1.4'),
            environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('startrek_client', version="2.3.0",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
        )

    def get_list(self, array):
        result = []
        parts = array.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return result

    def create_startrek_ticket(self, ticket_description):
        summary = 'Расхождения между ORDERDETAIL и PICKDETAIL'
        st_token = self.Parameters.st_token_secret.data()
        st_token_key = self.Parameters.st_token_secret.default_key or 'token'
        st_token = st_token[st_token_key]
        from startrek_client import Startrek
        st = Startrek(
            useragent=self.Parameters.st_user_agent,
            base_url=self.Parameters.st_url,
            token=st_token
        )

        st.issues.create(
            queue=self.Parameters.st_queue,
            summary=self.Parameters.st_summary + ' - ' + summary,
            type={'name': 'Technical Task'},
            description=ticket_description,
            tags=self.Parameters.st_tags
        )

    def wms_closed_connection(self, conn):
        conn.close()
        logging.info('WMS database connection closed.')

    def print_result_od(self, data):
        ticket_description = ''
        if data:
            ticket_description = '**Найдены расхождения между ORDERDETAIL и PICKDETAIL**\n'
            ticket_description += '#\n'
            ticket_description += '| orderkey | orderlinenumber | sku | type | od_qtyallocated | od_qtypicked |' \
                                  'pd_qtyallocated | pd_qtypicked ' \
                                 '|\n| :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: |'
            for orderkey, orderlinenumber, sku, type, od_qtyallocated, od_qtypicked, pd_qtyallocated, pd_qtypicked in data:
                ticket_description = ticket_description + '\n| {} | {} | {} | {} | ' \
                                                          '{} | {} | {} | {} |'.format(orderkey, orderlinenumber, sku,
                                                                                       type, od_qtyallocated,
                                                                                        od_qtypicked,
                                                                                        pd_qtyallocated,
                                                                                        pd_qtypicked)
            ticket_description += '\n#'
        return ticket_description

    def wms_request_od(self, conn):
        curr = conn.cursor()
        curr.execute(
            """
                select od.orderkey, od.ORDERLINENUMBER, od.sku, o.type,
                isnull(od.qtyallocated, 0) as [od_qtyallocated], isnull(od.qtypicked, 0) as [od_qtypicked],
                isnull(pda.qty, 0) as [pd_qtyallocated], isnull(pdp.qty, 0) as [pd_qtypicked]
                from SCPRD.wmwhse1.ORDERDETAIL od
                full outer join (select orderkey, orderlinenumber, sku, sum(qty) as qty
                from SCPRD.wmwhse1.pickdetail pd  where status between '0' and '4' group by orderkey, orderlinenumber, sku)
                pda
                on od.orderkey = pda.orderkey and od.ORDERLINENUMBER = pda.orderlinenumber
                full outer join (select orderkey, orderlinenumber, sku, sum(qty) as qty
                from SCPRD.wmwhse1.pickdetail pd where status between '5' and '8' group by orderkey, orderlinenumber, sku)
                pdp
                on od.orderkey = pdp.orderkey and od.ORDERLINENUMBER = pdp.orderlinenumber
                join SCPRD.wmwhse1.orders o on o.orderkey = od.orderkey
                where ( isnull(od.qtyallocated, 0) <> isnull(pda.qty, 0)
                OR isnull(od.qtypicked, 0) <> isnull(pdp.qty, 0))
                and o.type in ('0','100') and o.STATUS not in ('98','99')
            """
        )
        data = curr.fetchall()
        return data

    def wms_connection(self, host, port, user, password):
        conn = None
        logging.info('Connecting to the WMS database...')
        logging.info('User: {}'.format(user))
        _pymssql = __import__('pymssql')
        conn = _pymssql.connect(
            user=user,
            password=password,
            host=host,
            port=port
        )
        logging.info('Connected')
        return conn

    def on_execute(self):
        hosts = self.get_list(self.Parameters.database_host_string)
        ports = self.get_list(self.Parameters.database_port_string)
        warehouses = self.get_list(self.Parameters.database_warehouse_string)
        database_user = self.Parameters.database_user_secret.data()
        database_user_key = self.Parameters.database_user_secret.default_key or 'username'
        database_user = database_user[database_user_key]
        database_password = self.Parameters.database_password_secret.data()
        database_password_key = self.Parameters.database_password_secret.default_key or 'password'
        database_password = database_password[database_password_key]

        ticket_description = '((https://datalens.yandex-team.ru/exr93tjagoww6-rashozhdeniya-balansovyh-tablic?tab=zW' \
                             ' Дашборд с расхождениями))\n'
        ticket_description_tmp = ''

        for i in range(0, 8):
            counter = 0
            ticket_description_tmp += '<{' + warehouses[i] + '\n'
            conn = self.wms_connection(hosts[i], ports[i], database_user, database_password)
            result_od = self.wms_request_od(conn)
            od_size = len(result_od)
            if od_size > 0:
                ticket_description_tmp += self.print_result_od(result_od) + '\n'
                counter += 1
            ticket_description_tmp += '}>'
            logging.info(ticket_description_tmp)
            if counter > 0:
                ticket_description += ticket_description_tmp + '\n'
            ticket_description_tmp = ''
            self.wms_closed_connection(conn)
            self.wms_closed_connection(conn)
            logging.info(ticket_description)
        self.create_startrek_ticket(ticket_description)
