# coding=utf-8
import logging

from sandbox import sdk2
from sandbox.sdk2 import environments
from sandbox.common.telegram import TelegramBot


class WmsSupportEvaExecutor(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("WMS database settings", collapse=True) as wms_db_settings:
            database_user_secret = sdk2.parameters.YavSecret(
                'WMS database username'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: username',
                required=True
            )
            database_password_secret = sdk2.parameters.YavSecret(
                'WMS database password '
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: password',
                required=True
            )
            database_host_string = sdk2.parameters.String('host1, host2,...', required=True)
            database_port_string = sdk2.parameters.String('port1, port2,...', required=True)
            database_warehouse_string = sdk2.parameters.String('warehouse1, warehouse2,...', required=True)
            telegram_bot_token_vault_key = sdk2.parameters.String('Telegram bot token key', required=True)
            telegram_chat_id = sdk2.parameters.String('Telegram chat id', required=True)

        with sdk2.parameters.Group("Startrek settings", collapse=True) as st_settings:
            st_user_agent = sdk2.parameters.String('Startrek user agent', required=True)
            st_url = sdk2.parameters.String('Startrek url', required=True)
            st_token_secret = sdk2.parameters.YavSecret(
                'Startrek token'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: token',
                required=True
            )
            st_queue = sdk2.parameters.String('Startrek queue', required=True)
            st_summary = sdk2.parameters.String('Startrek ticket summary', required=True)
            st_tags = sdk2.parameters.List('List of tags for Startrek issue')
            board = sdk2.parameters.String('Board URL')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            environments.PipEnvironment('pymssql==2.1.4'),
            environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('startrek_client', version="2.3.0",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
        )

    def get_list(self, array):
        result = []
        parts = array.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return result

    def create_startrek_ticket(self, ticket_description):
        summary = 'Расхождения балансов'
        st_token = self.Parameters.st_token_secret.data()
        st_token_key = self.Parameters.st_token_secret.default_key or 'token'
        st_token = st_token[st_token_key]
        from startrek_client import Startrek
        st = Startrek(
            useragent=self.Parameters.st_user_agent,
            base_url=self.Parameters.st_url,
            token=st_token
        )

        st.issues.create(
            queue=self.Parameters.st_queue,
            summary=self.Parameters.st_summary + ' - ' + summary,
            type={'name': 'Technical Task'},
            description=ticket_description,
            tags=self.Parameters.st_tags
        )

    def wms_closed_connection(self, conn):
        conn.close()
        logging.info('WMS database connection closed.')

    def print_result_lot(self, data):
        ticket_description = ''
        counter = 0
        if data:
            ticket_description = '**Найдены расхождения между LOT и LOTXLOCXID**\n'
            ticket_description += '#\n'
            ticket_description += '| storerkey | sku | lot | l_qty | l_qtyallocated | l_qtypicked | lli_qty | ' \
                                'lli_qtyallocated | lli_qtypicked' \
                                '|\n| :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: |'
            for storerkey, sku, lot, l_qty, l_qtyallocated, l_qtypicked, lli_qty, lli_qtyallocated,\
                    lli_qtypicked in data:
                ticket_description = ticket_description + '\n| {} | {} | {} | {} | {} | {} | {} | {} | {} |'.format(
                    storerkey, sku, lot, l_qty, l_qtyallocated, l_qtypicked, lli_qty, lli_qtyallocated, lli_qtypicked)
            ticket_description += '\n#'
        return ticket_description

    def wms_request_lot(self, conn):
        curr = conn.cursor()
        curr.execute(
            """select l.storerkey as [storerkey], l.sku as [sku], l.lot as [lot],  isnull(l.qty, 0) as [l_qty],
               isnull(l.qtyallocated, 0) as [l_qtyallocated], isnull(l.qtypicked, 0) as [l_qtypicked],
               isnull(lli.qty, 0) as [lli_qty],
               isnull(lli.qtyallocated, 0) as [lli_qtyallocated], isnull(lli.qtypicked, 0) as [lli_qtypicked]
               from scprd.wmwhse1.lot l
               full outer join (select storerkey, sku, lot,  sum(qty) qty, sum(qtyallocated) qtyallocated,
               sum(qtypicked) qtypicked from scprd.wmwhse1.lotxlocxid group by storerkey, sku, lot) lli
               on l.storerkey = lli.storerkey and l.sku = lli.sku and l.lot = lli.lot
               where (isNull(l.qty, 0) <> isNull(lli.qty, 0)
               OR isNull(l.qtyallocated, 0) <> isNull(lli.qtyallocated, 0)
               OR isNull(l.qtypicked, 0) <> isNull(lli.qtypicked, 0))
               and lli.sku not in ('PL','BOX')
            """
        )
        data = curr.fetchall()
        return data

    def print_result_sl(self, data):
        ticket_description = ''
        if data:
            ticket_description = '**Найдены расхождения между SKUXLOC и LOTXLOCXID**\n'
            ticket_description += '#\n'
            ticket_description += '| storerkey | sku | loc | sl_qty | sl_qtyallocated | sl_qtypicked | lli_qty | ' \
                                'lli_qtyallocated | lli_qtypicked' \
                                '|\n| :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: |'
            for storerkey, sku, loc, sl_qty, sl_qtyallocated, sl_qtypicked, lli_qty, lli_qtyallocated,\
                    lli_qtypicked in data:
                ticket_description = ticket_description + '\n| {} | {} | {} | {} | {} | ' \
                                                          '{} | {} | {} | {} |'.format(storerkey, sku, loc, sl_qty,
                                                                                       sl_qtyallocated, sl_qtypicked,
                                                                                       lli_qty, lli_qtyallocated,
                                                                                       lli_qtypicked)
            ticket_description += '\n#'
        return ticket_description

    def wms_requset_sl(self, conn):
        curr = conn.cursor()
        curr.execute(
            """select sl.storerkey as [storerkey], sl.sku as [sku], sl.loc as [loc], isnull(sl.qty, 0) as [sl_qty],
               isnull(sl.qtyallocated, 0) as [sl_qtyallocated], isnull(sl.qtypicked, 0) as [sl_qtypicked],
               isnull(lli.qty, 0) as [lli_qty],
               isnull(lli.qtyallocated, 0) as [lli_qtyallocated], isnull(lli.qtypicked, 0) as [lli_qtypicked]
               from scprd.wmwhse1.skuxloc sl
               full outer join (select storerkey, sku, loc, sum(qty) qty, sum(qtyallocated) qtyallocated,
               sum(qtypicked) qtypicked from scprd.wmwhse1.lotxlocxid group by storerkey, sku, loc) lli
               on sl.storerkey = lli.storerkey and sl.sku = lli.sku and sl.loc = lli.loc
               where ((isNull(sl.qty, 0) <> isNull(lli.qty, 0)
               OR isNull(sl.qtyallocated, 0) <> isNull(lli.qtyallocated, 0)
               OR isNull(sl.qtypicked, 0) <> isNull(lli.qtypicked, 0)))
               and lli.sku not in ('PL','BOX')
            """
        )
        data = curr.fetchall()
        return data

    def print_result_lli(self, data):
        ticket_description = ''
        if data:
            ticket_description = '**Найдены расхождения между LOTXLOCXID и PICKDETAIL**\n'
            ticket_description += '#\n'
            ticket_description += '| storerkey | sku | lot | loc | qty | id | lli_qtyallocated | lli_qtypicked |' \
                                 ' pd_qtyallocated | pd_qtypicked ' \
                                 '|\n| :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: | :---: |'
            for storerkey, sku, lot, loc, qty, id, lli_qtyallocated, lli_qtypicked, pd_qtyallocated,\
                    pd_qtypicked in data:
                ticket_description = ticket_description + '\n| {} | {} | {} | {} | ' \
                                                          '{} | {} | {} | {} | {} | {} |'.format(
                                                                                                storerkey, sku, lot,
                                                                                                loc, qty, id,
                                                                                                lli_qtyallocated,
                                                                                                lli_qtypicked,
                                                                                                pd_qtyallocated,
                                                                                                pd_qtypicked)
            ticket_description += '\n#'
        return ticket_description

    def wms_request_lli(self, conn):
        curr = conn.cursor()
        curr.execute(
            """
                select lli.storerkey as [storerkey], lli.sku as [sku], lli.lot as [lot], lli.loc as [loc],
                lli.qty as [qty], lli.id as [id],
                isnull(lli.qtyallocated, 0) as [lli_qtyallocated], isnull(lli.qtypicked, 0) as [lli_qtypicked],
                isnull(pda.qty, 0) as [pd_qtyallocated], isnull(pdp.qty, 0) as [pd_qtypicked]
                from scprd.wmwhse1.lotxlocxid lli
                full outer join (select storerkey, sku, lot, loc, id, sum(qty) as qty from scprd.wmwhse1.pickdetail pd
                where status between '0' and '4' group by storerkey, sku, lot, loc, id) pda
                on lli.storerkey = pda.storerkey and lli.lot = pda.lot and lli.loc = pda.loc and lli.id = pda.id
                full outer join (select storerkey, sku, lot, loc, id, sum(qty) as qty from scprd.wmwhse1.pickdetail pd
                where status between '5' and '8' group by storerkey, sku, lot, loc, id) pdp
                on lli.storerkey = pdp.storerkey and lli.lot = pdp.lot and lli.loc = pdp.loc and lli.id = pdp.id
                where (( isnull(lli.qtyallocated, 0) <> isnull(pda.qty, 0)
                OR isnull(lli.qtypicked, 0) <> isnull(pdp.qty, 0)))
                and lli.sku not in ('PL','BOX')
            """
        )
        data = curr.fetchall()
        return data

    def wms_connection(self, host, port, user, password):
        conn = None
        logging.info('Connecting to the WMS database...')
        logging.info('User: {}'.format(user))
        _pymssql = __import__('pymssql')
        conn = _pymssql.connect(
            user=user,
            password=password,
            host=host,
            port=port
        )
        logging.info('Connected')
        return conn

    def on_execute(self):
        bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
        hosts = self.get_list(self.Parameters.database_host_string)
        ports = self.get_list(self.Parameters.database_port_string)
        warehouses = self.get_list(self.Parameters.database_warehouse_string)
        database_user = self.Parameters.database_user_secret.data()
        database_user_key = self.Parameters.database_user_secret.default_key or 'username'
        database_user = database_user[database_user_key]
        database_password = self.Parameters.database_password_secret.data()
        database_password_key = self.Parameters.database_password_secret.default_key or 'password'
        database_password = database_password[database_password_key]

        ticket_description = '((https://datalens.yandex-team.ru/exr93tjagoww6-rashozhdeniya-balansovyh-tablic?tab=zW' \
                             ' Дашборд с расхождениями))\n'
        tg_message = ''
        tg_message_tmp = ''

        for i in range(0, 8):
            counter = 0
            ticket_description += '<{' + warehouses[i] + '\n'
            tg_message_tmp += 'На складе '+ warehouses[i] + ' найдено:\n'
            conn = self.wms_connection(hosts[i], ports[i], database_user, database_password)
            result_lli = self.wms_request_lli(conn)
            lli_size = len(result_lli)
            if lli_size > 0:
                tg_message_tmp += str(lli_size) + '   расхождений между LOTXLOCXID и PICKDETAIL\n'
                counter += 1
                ticket_description += self.print_result_lli(result_lli)
                ticket_description += '\n'
            #logging.info(tg_message_tmp)
            #logging.info('to tg  ', tg_message)
            result_sl = self.wms_requset_sl(conn)
            sl_size = len(result_sl)
            if sl_size > 0:
                tg_message_tmp += str(sl_size) + '   расхождений между SKUXLOC и LOTXLOCXID\n'
                counter += 1
                ticket_description += self.print_result_sl(result_sl)
                ticket_description += '\n'
            #logging.info(tg_message_tmp)
            #logging.info('to tg  ', tg_message)
            result_lot = self.wms_request_lot(conn)
            lot_size = len(result_lot)
            if lot_size > 0:
                tg_message_tmp += str(lot_size) + '   расхождений между LOT и LOTXLOCXID\n'
                counter += 1
                ticket_description += self.print_result_lot(result_lot)
                ticket_description += '\n'
            ticket_description += '}>'
            logging.info(tg_message_tmp)
            #logging.info('to tg  ', tg_message)
            if counter > 0:
                tg_message += tg_message_tmp + '\n'
            tg_message_tmp = ''
            self.wms_closed_connection(conn)
        logging.info('to tg  ', tg_message)
        bot.send_message(self.Parameters.telegram_chat_id, tg_message)
            #logging.info(ticket_description)
        #self.create_startrek_ticket(ticket_description)
