# -*- coding: utf-8 -*-
import logging

from sandbox import sdk2
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments


class WmsSupportCheckRoleSecond(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        database_user_secret = sdk2.parameters.YavSecret(
            'WMS database username'
            '\nYAV secret identifier (with optional version)'
            '\nDefault key: username',
            required=True
        )
        database_password_secret = sdk2.parameters.YavSecret(
            'WMS database password '
            '\nYAV secret identifier (with optional version)'
            '\nDefault key: password',
            required=True
        )
        database_host_string = sdk2.parameters.String('host1, host2,...', required=True)
        database_port_string = sdk2.parameters.String('port1, port2,...', required=True)
        database_warehouse_string = sdk2.parameters.String('warehouse1, warehouse2,...', required=True)
        telegram_bot_token_vault_key = sdk2.parameters.String('Telegram bot token key', required=True)
        telegram_chat_id = sdk2.parameters.String('Telegram chat id', required=True)

        with sdk2.parameters.Group("Startrek settings", collapse=True) as st_settings:
            st_user_agent = sdk2.parameters.String('Startrek user agent', required=True)
            st_url = sdk2.parameters.String('Startrek url', required=True)
            st_token_secret = sdk2.parameters.YavSecret(
                'Startrek token'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: token',
                required=True
            )
            st_queue = sdk2.parameters.String('Startrek queue', required=True)
            st_summary = sdk2.parameters.String('Startrek ticket summary', required=True)
            st_tags = sdk2.parameters.List('List of tags for Startrek issue')
            board = sdk2.parameters.String('Board URL')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('pymssql==2.1.4'),
                        environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]),
                        environments.PipEnvironment('startrek_client', version="2.3.0",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]))

    def get_list(self, array):
        result = []
        parts = array.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return result

    def create_startrek_ticket(self, ticket_description):
        summary = 'Incorrect role'
        st_token = self.Parameters.st_token_secret.data()
        st_token_key = self.Parameters.st_token_secret.default_key or 'token'
        st_token = st_token[st_token_key]
        from startrek_client import Startrek
        st = Startrek(
            useragent=self.Parameters.st_user_agent,
            base_url=self.Parameters.st_url,
            token=st_token
        )

        st.issues.create(
            queue=self.Parameters.st_queue,
            summary=self.Parameters.st_summary + ' - ' + summary,
            type={'name': 'Technical Task'},
            description=ticket_description,
            tags=self.Parameters.st_tags
        )

    def wms_closed_connection(self, conn):
        conn.close()
        logging.info('WMS database connection closed.')

    def wms_request_role(self, conn, warehouse):
        curr = conn.cursor()
        curr.execute(
            """SELECT externuserid, sso_role_name
            FROM        SCPRDD1.dbo.e_sso_user_role AS ESUL
            INNER JOIN  SCPRDD1.dbo.e_sso_role      AS ESR
            ON ESUL.sso_role_id=ESR.e_sso_role_id
            where sso_role_name like '%SCE-Administrator%'"""
        )
        row = curr.fetchone()
        data = ''
        while row:
            data += 'User   ' + row[0] + '   has role   ' + row[1] + '   on    ' + warehouse + '\n'
            logging.info('Data: {}'.format(data))
            row = curr.fetchone()
        return data

    def wms_connection(self, host, port, user, password):
        conn = None
        logging.info('Connecting to the WMS database...')
        logging.info('User: {}'.format(user))
        _pymssql = __import__('pymssql')
        conn = _pymssql.connect(
            user=user,
            password=password,
            host=host,
            port=port
        )
        logging.info('Connected')
        return conn

    def on_execute(self):
        bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
        hosts = self.get_list(self.Parameters.database_host_string)
        ports = self.get_list(self.Parameters.database_port_string)
        warehouses = self.get_list(self.Parameters.database_warehouse_string)
        database_user = self.Parameters.database_user_secret.data()
        database_user_key = self.Parameters.database_user_secret.default_key or 'username'
        database_user = database_user[database_user_key]
        database_password = self.Parameters.database_password_secret.data()
        database_password_key = self.Parameters.database_password_secret.default_key or 'password'
        database_password = database_password[database_password_key]
        data = ''
        for i in range(0, 8):
            conn = self.wms_connection(hosts[i], ports[i], database_user, database_password)
            data += self.wms_request_role(conn, warehouses[i])
            self.wms_closed_connection(conn)
        logging.info(data)
        if data:
            bot.send_message(self.Parameters.telegram_chat_id, data)
            self.create_startrek_ticket(data)
