import json
import logging
import os
import requests
import time

import sandbox.common.types.task as ctt

from collections import namedtuple
from sandbox import sdk2


AccessResource = namedtuple('AccessResource', [
    'name',
    'version',
])


class MarketAccessResource(sdk2.Resource):
    ttl = 15


# type: MARKET_ACCESS_PLAY_RESOURCE_SUBTASK
class MarketAccessPlayResourceSubtask(sdk2.Task):
    _ALPHA_FILE_SIZE_BYTES = 256 * 1024  # 256 KiB
    _BETA_FILE_SIZE_BYTES = 512 * 1024  # 512 KiB
    _GAMMA_FILE_SIZE_BYTES = 1024 * 1024  # 1 MiB

    class Requirements(sdk2.Requirements):
        disk_space = 1000

    class Parameters(sdk2.Parameters):
        alpha_resource = sdk2.parameters.ParentResource('alpha', required=True)
        beta_resource = sdk2.parameters.ParentResource('beta', required=True)
        gamma_resource = sdk2.parameters.ParentResource('gamma', required=True)

    def on_execute(self):
        logging.info('Let\'s prepare resources')
        self._initialize_resource(self.Parameters.alpha_resource, 'alpha.res', self._ALPHA_FILE_SIZE_BYTES)
        self._initialize_resource(self.Parameters.beta_resource, 'beta.res', self._BETA_FILE_SIZE_BYTES)
        self._initialize_resource(self.Parameters.gamma_resource, 'gamma.res', self._GAMMA_FILE_SIZE_BYTES)
        logging.info('All resources are ready for use')

    @classmethod
    def _initialize_resource(cls, resource, filename, size_bytes):
        file_path = cls._generate_file(filename, size_bytes)
        cls._write_resource_content(resource, file_path)

    @staticmethod
    def _generate_file(name, size_bytes):
        logging.info('Will generate a new file "%s" with size %d bytes', name, size_bytes)
        path = os.path.join(os.getcwd(), name)
        with open(path, 'wb') as fout:
            fout.write(os.urandom(size_bytes))
        logging.info('New file with size %d bytes was generated, path: %s', size_bytes, path)
        return path

    @staticmethod
    def _write_resource_content(resource, filename):
        logging.info('Writing content of resource "%s"', resource.description)
        with open(filename, 'rb') as f:
            resource.path.write_bytes(f.read())


# type: MARKET_ACCESS_PLAY_VERSION_PUBLISHER
class MarketAccessPlayVersionPublisher(sdk2.Task):
    _SUBTASK_WAIT_TIMEOUT_SECONDS = 300

    _PARENT_RESOURCE_NAME = 'alpha'
    _DEPENDANT_RESOURCE_NAMES = ['beta', 'gamma']
    _RESOURCE_NAMES = [_PARENT_RESOURCE_NAME] + _DEPENDANT_RESOURCE_NAMES

    _ACCESS_SERVER_URL = 'https://access.tst.vs.market.yandex.net'
    _RETRIES_PER_RESOURCE = 10
    _TIMEOUT_BETWEEN_RETRIES_SECONDS = 1

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True

    class Context(sdk2.Context):
        resource_subtask_id = None
        resource_ids = dict()

    def on_execute(self):
        logging.info('Let\'s update some versions for access_play')
        resources = self._acquire_resources()

        for resource_name, resource in resources.items():
            logging.info('Got resource "%s" with id: %d and rbtorrent: %s', resource_name, resource.id, resource.skynet_id)

        parent_resource = self._publish_version(self._PARENT_RESOURCE_NAME, resources[self._PARENT_RESOURCE_NAME])
        for resource_name in self._DEPENDANT_RESOURCE_NAMES:
            self._publish_version(resource_name, resources[resource_name], dependency=parent_resource)

    @classmethod
    def _publish_version(cls, resource_name, resource, dependency=None):
        def is_ok(response):
            return response.status_code == requests.codes.ok \
                and response.json().get('status') != 'error'

        path = '{}/v1/resources/{}/versions'.format(cls._ACCESS_SERVER_URL, resource_name)
        body = {
            'resource_name': resource_name,
            'storage': [{
                'location': {
                    'access': {
                        'rbtorrent': resource.skynet_id,
                    }
                }
            }]
        }
        if dependency:
            body['dependency'] = [{
                'resource_name': dependency.name,
                'version_number': dependency.version,
            }]
        body = json.dumps(body)

        for _ in range(cls._RETRIES_PER_RESOURCE):
            response = requests.post(path, data=body)
            if is_ok(response):
                as_json = response.json()
                logging.info('Resource "%s" was published successfully', resource_name)
                return AccessResource(name=resource_name, version=as_json['number'])
            time.sleep(cls._TIMEOUT_BETWEEN_RETRIES_SECONDS)

        logging.error('Publishing error: %s', json.dumps(response.json()))
        raise RuntimeError('Unable to publish resource "{}"'.format(resource_name))

    def _acquire_resources(self):
        if self.Context.resource_subtask_id is not None:
            return self._list_resources()
        resource_params = dict()
        for resource_name in self._RESOURCE_NAMES:
            filename = resource_name + '.res'
            resource = MarketAccessResource(self, resource_name, filename)
            self.Context.resource_ids[resource_name] = resource.id
            resource_params[resource_name + '_resource'] = resource
        resource_subtask = MarketAccessPlayResourceSubtask(
            self,
            description='Child of {}'.format(self.id),
            owner=self.owner,
            **resource_params
        )
        resource_subtask.enqueue()
        self.Context.resource_subtask_id = resource_subtask.id
        raise sdk2.WaitTask(
            [resource_subtask],
            ctt.Status.Group.FINISH,
            wait_all=True,
            timeout=self._SUBTASK_WAIT_TIMEOUT_SECONDS,
        )

    def _list_resources(self):
        self._check_resource_subtask()
        resources = dict()
        for resource_name, resource_id in self.Context.resource_ids.items():
            resource = sdk2.Resource.find(id=resource_id).first()
            if resource is None:
                raise RuntimeError('Resource "{}" with id {} doesn\'t exist', resource_name, resource_id)
            resources[resource_name] = resource
        return resources

    def _check_resource_subtask(self):
        task_id = self.Context.resource_subtask_id
        task = MarketAccessPlayResourceSubtask.find(id=task_id, children=True).first()
        if task is None:
            raise RuntimeError('Resource task #{} does\'t exist'.format(task_id))
        if task.status != ctt.Status.SUCCESS:
            raise RuntimeError('Resource task #{} have failed with status: {}'.format(
                task_id,
                str(task.status),
            ))
