# coding=utf-8

import json
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class MarketAffiliateDistributionPartners(sdk2.Task):
    """
    Таска для забора информации о партнерах, участниках партнерской сети
    Яндекс.Маркета, из справочника выгружаемого Яндекс.Дистрибуцией

    Ходит в YT в табличку https://nda.ya.ru/3UrRAz
    Забирает данные, и перекладывает их в S3
    """

    YT_PROXY = "hahn"
    YT_DIC_PATH = "//home/market/production/mstat/dictionaries"
    YT_TABLE_PATH = YT_DIC_PATH + "/distribution_partners/latest"

    S3_HOST = "s3.mds.yandex.net"
    S3_BUCKET_NAME = "market-affiliate"
    S3_OBJECT_NAME = "distribution-partners.json"

    @property
    def YT_TOKEN(self):
        return sdk2.Vault.data("robot-metatron-yt-token")

    @property
    def AWS_ACCESS_KEY_ID(self):
        return sdk2.Vault.data("robot-metatron-s3-keyid")

    @property
    def AWS_SECRET_ACCESS_KEY(self):
        return sdk2.Vault.data("robot-metatron-s3-secretkey")

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("boto"),
            environments.PipEnvironment("yandex-yt"),
        ]

    def on_execute(self):
        super(MarketAffiliateDistributionPartners, self).on_execute()

        data = self._read_distribution_partners_info()

        self._write_distribution_partners_info(data)

    def _get_yt_client(self):
        from yt.wrapper import YtClient

        return YtClient(proxy=self.YT_PROXY, token=self.YT_TOKEN)

    def _get_s3_connection(self):
        from boto.s3.connection import S3Connection

        return S3Connection(
            host=self.S3_HOST,
            aws_access_key_id=self.AWS_ACCESS_KEY_ID,
            aws_secret_access_key=self.AWS_SECRET_ACCESS_KEY,
        )

    def _read_distribution_partners_info(self):
        from yt.wrapper import JsonFormat

        client = self._get_yt_client()
        rows_iterator = client.read_table(
            table=self.YT_TABLE_PATH,
            format=JsonFormat(attributes={"encode_utf8": False})
        )

        result = []

        for row in rows_iterator.rows:
            item = {}
            item["clid"] = row["clid"]
            item["clidTypeId"] = row["clid_type_id"]
            item["userLogin"] = row["user_login"]
            item["cpcRevShare"] = row["cpc_rev_share"]
            item["cpaRevShare"] = row["cpa_rev_share"]
            item["packDomain"] = row["pack_domain"]
            item["packId"] = row["pack_id"]
            item["softId"] = row["soft_id"]
            result.append(item)

        return result

    def _write_distribution_partners_info(self, data):
        conn = self._get_s3_connection()

        bucket = conn.get_bucket(self.S3_BUCKET_NAME)
        key = bucket.new_key(self.S3_OBJECT_NAME)
        key.set_contents_from_string(json.dumps(data, ensure_ascii=False))
