# -*- coding: utf-8 -*-

"""
@author: alexahdp@yandex-team.ru
Модуль парсинга вакансий из бункера
Подробная информация: https://wiki.yandex-team.ru/users/alexahdp/Asylum
"""

import requests
import json
import time
import logging
import sandbox.projects.market.front.helpers.s3 as s3

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv


class MarketAsylum(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        # environments = (environments.PipEnvironment('boto3'),)
        environments = (environments.PipEnvironment('boto'),)

    class Parameters(sdk2.Task.Parameters):
        bucket_name = sdk2.parameters.String(
            "Bucket's name",
            default_value="marketfront"
        )

        filename = sdk2.parameters.String(
            "Filename",
            default_value="hr-digital-dev"
        )

    def on_execute(self):
        with MetatronEnv(self):
            time_start = time.time()
            parser = Parser()

            # res = parser.get_departments()
            departments = ['back_office']

            result = {}
            for department in departments:
                department_vacancies = parser.get_department_vacancies(department)
                if not department_vacancies:
                    logging.info('department_vacancies is None')

                result[department] = {}
                for vacancy_short_info in department_vacancies[0:3]:
                    vacancy = parser.get_vacancy(department, vacancy_short_info['name'])
                    result[department][vacancy_short_info['name']] = vacancy

                    if vacancy is not None:
                        result[department][vacancy_short_info['name']]['form_json'] = parser.get_form(vacancy['formId'])

            time_end = time.time()
            evaluation_time = time_end - time_start
            logging.info('evaluation time: {}ms'.format(evaluation_time))

            conn = s3.get_connection()
            s3.safe_upload_data(conn, self.Parameters.bucket_name, json.dumps(result, ensure_ascii=False).encode('utf8'), self.Parameters.filename)

            logging.info('end')


class Parser:
    def build_url(self, method='ls', path=[]):
        return 'http://bunker-api-dot.yandex.net/v1/' + method + '?node=vacancies-www/vacancies/' + '/'.join(path)

    def build_form_url(self, form_id):
        return 'https://forms-ext-api.yandex.ru/v1/surveys/{}/form/'.format(form_id)

    def get_departments(self):
        try:
            return self.fetch_json(self.build_url('/'))
        except Exception as e:
            logging.error('get_department_vacancies error: %s', {'exception': str(e)})
            return None

    def get_department_vacancies(self, dep_name):
        try:
            return self.fetch_json(self.build_url(path=[dep_name]))
        except Exception as e:
            logging.error('get_department_vacancies error: %s', {'dep_name': dep_name, 'exception': str(e)})
            return None

    def get_vacancy(self, dep_name, vacancy_name):
        try:
            return self.fetch_json(self.build_url(method='cat', path=[dep_name, vacancy_name]))
        except Exception as e:
            logging.error('get_vacancy error: %s', {'dep_name': dep_name, 'vacancy_name': vacancy_name, 'exception': str(e)})
            return None

    def fetch_json(self, url):
        response = requests.get(url)
        response.raise_for_status()
        json = response.json()
        return json

    def get_form(self, form_id):
        try:
            return self.fetch_json('https://forms-ext-api.yandex.ru/v1/surveys/{}/form/'.format(form_id))
        except:
            logging.error('Load form fail: %s', {'form_id': form_id})
            return None
