# coding=UTF8
import requests
import logging
import json
from datetime import datetime
from dateutil import parser
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi


class Escalator:

  def __init__(self,
               yav_tokens,
               calendar_id,
               transition_hour,
               notify_filter_id,
               stuck_filter_id,
               escalation_filter_id,
               escalation_enabled,
               reminder_enabled,
               stuck_enabled,
               start_weekday,
               end_weekday,
               start_hour,
               end_hour,
               service_responsible):
    self.startrek_oauth_token = yav_tokens.data()['startrek_oauth_token']
    self.telegram_bot_token = yav_tokens.data()['telegram_bot_token']
    self.abc_token = yav_tokens.data()['abc_token']
    self.startrek_api_url = 'https://st-api.yandex-team.ru/v2'
    self.telegram_api_url = 'https://api.telegram.org'
    self._staff = StaffApi(self.startrek_oauth_token)
    self.logger = logging.getLogger(__name__)
    self.abc_api_url = 'https://abc-back.yandex-team.ru/api/v4/duty/shifts/?service__slug=checkouter&date_from={}&date_to={}&with_watcher=1'
    self.alert_instruction_url = 'https://wiki.yandex-team.ru/market/marketplace/dev/dezhurstva/how-to-work-with-checkouter-alerts'
    self.startrek_issues_url = 'https://st.yandex-team.ru/issues/'

    self.calendar_id = calendar_id
    self.transition_hour = transition_hour
    self.notify_filter_id = notify_filter_id
    self.stuck_filter_id = stuck_filter_id
    self.escalation_filter_id = escalation_filter_id

    self.escalation_enabled = escalation_enabled
    self.reminder_enabled = reminder_enabled
    self.stuck_enabled = stuck_enabled

    # Schedule
    self.start_weekday = start_weekday
    self.end_weekday = end_weekday
    self.start_hour = start_hour
    self.end_hour = end_hour

    self.service_responsible = service_responsible

  def do_escalation(self):
    hour = datetime.now().time().hour
    day = datetime.now().weekday()
    if self.start_weekday <= day <= self.end_weekday and self.start_hour <= hour <= self.end_hour:
      if self.reminder_enabled:
        self.notify_duty()
      if self.escalation_enabled:
        self.escalate_alerts()
      if self.stuck_enabled:
        self.remind_in_progress()

  def notify_duty(self):
    issues = self.get_issues(self.notify_filter_id)
    self.logger.info("Found issues to remind {}".format(issues))
    if issues:
      message = "Алертные тикеты, оставшиеся, без внимания! " \
                "Пожалуйста, отработайте их в соответствтвии с <a href=\"{}\">инструкцией</a>" \
        .format(self.alert_instruction_url)

      message = self.add_issues_to_message(message, issues)
      if len(issues) > 1:
          message = message + "\n" + ("<a href=\"{}{}\">Список в трекере</a>".format(self.startrek_issues_url, self.notify_filter_id))

      self.send_telegram_message(message)

  def escalate_alerts(self):
    issues = self.get_issues(self.escalation_filter_id)
    self.logger.info("Found issues to escalate {}".format(issues))

    if issues:
      duty = self.get_current_duty()
      message = "Внимание! @{} @{}\nСписок тикетов, с нарушением SLA на время первой реакции:".format(
        self.service_responsible,
        self.get_telegram_contact(duty)
      )

      message = self.add_issues_to_message(message, issues)
      if len(issues) > 1:
          message = message + "\n" + ("<a href=\"{}{}\">Список в трекере</a>".format(self.startrek_issues_url, self.escalation_filter_id))

      self.send_telegram_message(message)

  def remind_in_progress(self):
    issues = self.get_issues(self.stuck_filter_id)
    self.logger.info("Found stuck in progress issues {}".format(issues))
    if issues:
      message = self.add_issues_to_message("Тикеты зависшие в статусе \"В работе\"", issues)
      if len(issues) > 1:
        message = message + "\n" + ("<a href=\"{}{}\">Список в трекере</a>".format(self.startrek_issues_url, self.stuck_filter_id))
      self.send_telegram_message(message)

  def get_issues(self, filter_id):
    return requests.get(
      '{}/issues?query=Filter:{}'.format(self.startrek_api_url, filter_id),
      headers=self.startrek_headers()
    ).json()

  def get_current_duty(self):
    today = datetime.today().replace(hour=0, minute=0, second=0, microsecond=0)
    duties = requests.get(self.abc_api_url.format(today.strftime("%Y-%m-%d"), today.strftime("%Y-%m-%d")),
                          headers={'Authorization': 'OAuth {}'.format(self.abc_token)}).json()['results']
    self.logger.info("Duties found: {}".format(duties))

    duty_login = ''
    for duty in duties:
        start_datetime = parser.parse(duty['start_datetime'])
        end_datetime = parser.parse(duty['end_datetime'])
        now = datetime.now(start_datetime.tzinfo)
        if self.calendar_id == duty['schedule']['id'] and start_datetime < now < end_datetime:
            duty_login = duty['person']['login']

    return duty_login

  def get_telegram_contact(self, login):
    contacts = self._staff.get_person_profile_property(login, 'telegram_accounts')['telegram_accounts']
    for contact in contacts:
      if 'bot' not in contact['value']:
        return contact['value']

  def send_telegram_message(self, message):
    result = requests.post(
      '{}/bot{}/sendMessage'.format(self.telegram_api_url, self.telegram_bot_token),
      headers=self.startrek_headers(),
      data=json.dumps({
        "chat_id": "-1001324412277",
        "text": message,
        "parse_mode": "HTML",
        "method": "sendMessage"
      })
    ).json()
    self.logger.info("Sending notification result {}".format(result))

  def get_current_duty_manager(self, duty_login):
    return self._staff.get_person_profile_property(duty_login, 'chief')['chief']['login']

  def add_issues_to_message(self, message, issues):
    issue_text_pattern_with_assignee = "<a href=\"https://st.yandex-team.ru/{}\">{}</a>, ответственный @{}"
    issue_text_pattern = "<a href=\"https://st.yandex-team.ru/{}\">{}</a>"

    for issue in issues:
      if "assignee" in issue:
        message = message \
                  + "\n" \
                  + issue_text_pattern_with_assignee.format(issue["key"],  issue["key"],
                                              self.get_telegram_contact(issue["assignee"]['id']))
      else:
        message = message \
                  + "\n" \
                  + issue_text_pattern.format(issue["key"],  issue["key"])

    return message

  def startrek_headers(self):
    return {'Authorization': 'Bearer {}'.format(self.startrek_oauth_token), 'Content-Type': 'application/json'}
