#! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# https://sandbox.yandex-team.ru/oauth/
# ./MarketCarterLoadCalculateRps upload --verbose --owner MARKET --attr release=stable --attr ttl=inf --attr task_type=MARKET_CARTER_LOAD_CALCULATE_RPS

import codecs
import csv
import logging as log
import random

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from sandbox.projects.market.checkout.helpers import RTC
from sandbox.projects.market.checkout.helpers import YQL
from sandbox.projects.market.checkout.resources import CarterLoadEndpointsRps

random.seed(123)


class MarketCarterLoadCalculateRps(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yql_token_owner = sdk2.parameters.String('YQL token OWNER in Vault', default_value='zomb-emily')
        yql_token_name = sdk2.parameters.String('YQL token NAME in Vault', default_value='YQL_TOKEN')

        with sdk2.parameters.Output:
            prod_to_load_instance_count_ratio = sdk2.parameters.Float('prod-to-load instance count ratio',
                                                                      default_value='')
            prod_to_load_instance_power_ratio = sdk2.parameters.Float('prod-to-load instance power ratio',
                                                                      default_value='')
            endpoints_rps = sdk2.parameters.String('Endpoints and RPS (csv)', default_value='')

    def on_execute(self):
        log.info('Starting on_execute')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name)
        rps_query_resource_path = 'sandbox/projects/market/checkout/MarketCarterLoadCalculateRps/rps_request.sql'
        current_rps_total, rps_ratio = self.get_total_rps_and_ratios(yql_token, rps_query_resource_path)
        log.info('Total current RPS: ' + str(current_rps_total))
        log.info('RPS ratio result: ' + str(rps_ratio))
        instance_count_ratio, instance_power_ratio = self.get_prod_to_load_ratios()
        self.Parameters.prod_to_load_instance_count_ratio = instance_count_ratio
        self.Parameters.prod_to_load_instance_power_ratio = instance_power_ratio
        self.publish_output_resource(rps_ratio)

    def get_total_rps_and_ratios(self, yql_token, rps_query_resource_path):
        rows = YQL.query_clickhouse(yql_token, rps_query_resource_path)
        log.debug('got rps rows: {}'.format(rows))
        sum_rps = sum(map(lambda r: float(r[3]), rows))
        sum_parts = 0
        log.info('current RPS total:' + str(sum_rps))
        rps_ratio = []
        for row in rows:
            rps = float(row[3])
            rps_part = rps / sum_rps
            sum_parts += rps_part
            http_method = row[0]
            method = row[1]
            rgb = self.protect_from_request_param_duplication(row[2])
            rps_ratio.append([http_method, method, rgb, rps_part])
        log.debug('sum of all ratios:' + str(sum_parts))
        return sum_rps, rps_ratio

    def protect_from_request_param_duplication(self, rgb):
        if not rgb:
            return rgb
        # sometimes it can look like '[WHITE, WHITE]' because of request param duplication
        return rgb.replace('[', '').replace(']', '').replace(' ', '').split(',')[0]

    def get_prod_to_load_ratios(self):
        prod_category_prefix = 'market_carter_production'
        prod_service_prefix = 'production_market_carter'
        prod_services = RTC.get_services(category_prefix=prod_category_prefix, service_prefix=prod_service_prefix)
        prod_rtc_data = RTC.get_rtc_data(prod_services, prod_category_prefix)
        log.info('Prod config: %f cores, %f memory' % prod_rtc_data)
        dc_minus_1_prod_instance_count = RTC.get_dc_minus_1_prod_instance_count(services=prod_services,
                                                                                category_prefix=prod_category_prefix)
        log.info('DC-1 PROD instance count: {}'.format(dc_minus_1_prod_instance_count))
        load_category_prefix = 'market_carter_load'
        load_service_prefix = 'testing_market_carter_load'
        load_services = RTC.get_services(category_prefix=load_category_prefix, service_prefix=load_service_prefix)
        load_rtc_data = RTC.get_rtc_data(load_services, load_category_prefix)
        log.info('Load config: %f cores, %f memory' % load_rtc_data)
        instance_power_ratio = min(1 / (prod_rtc_data[0] / load_rtc_data[0]), 1 / (prod_rtc_data[1] / load_rtc_data[1]))
        log.info('instance_power_ratio= ' + str(instance_power_ratio))
        return dc_minus_1_prod_instance_count, instance_power_ratio

    def publish_output_resource(self, rps_ratio):
        data = sdk2.ResourceData(CarterLoadEndpointsRps(self, 'Carter load endpoints RPS', 'endpoints_rps.csv'))
        with codecs.open('endpoints_rps.csv', 'w', encoding='utf-8') as f:
            writer = csv.writer(f)
            for endpoint_rps in rps_ratio:
                writer.writerow(endpoint_rps)
        with open('endpoints_rps.csv', 'r') as f:
            self.Parameters.endpoints_rps = f.read()
        data.ready()
