#! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# ./MarketCarterLoadGenerateAmmo upload --verbose --owner MARKET --attr release=stable --attr ttl=inf --attr task_type=MARKET_CARTER_LOAD_GENERATE_AMMO

import codecs
import json
import logging as log
import random
import string
import uuid as uuidm

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

import constants
import resource_loader
from data_base import DataBase
from generate import make_ammo
from sandbox.projects.market.checkout.resources import CarterLoadAmmo
from sandbox.projects.market.checkout.resources import CarterLoadConfig
from sandbox.projects.market.checkout.resources import CarterLoadEndpointsRps

MAX_AMMO_COUNT = 1000000

random.seed(123)


class MarketCarterLoadGenerateAmmo(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yql_token_owner = sdk2.parameters.String('YQL token OWNER in Vault', default_value='zomb-emily')
        yql_token_name = sdk2.parameters.String('YQL token NAME in Vault', default_value='YQL_TOKEN')
        imbalance_shooting = sdk2.parameters.Bool(
            'Shoot until service is unstable?',
            description='When checked "line" tank rps schedule will be used until service fails. '
                        'When unchecked then rps will be constant ( "const" schedule )', default=True
        )
        with imbalance_shooting.value[False]:
            test_duration_minutes = sdk2.parameters.Integer('Test duration minutes', default=10)
            total_rps = sdk2.parameters.Integer('Total rps', default=8)

        with imbalance_shooting.value[True]:
            start_rps = sdk2.parameters.Integer('Start RPS', default=5)
            target_rps = sdk2.parameters.Integer('Target RPS', default=50)
            rps_increase_step = sdk2.parameters.Integer('RPS increase step', default=1)
            rps_step_duration_seconds = sdk2.parameters.Integer('RPS step duration seconds', default=100)
            autostop_config = sdk2.parameters.String('Shooting autostop config',
                                                     default=constants.DEFAULT_AUTOSTOP_CONFIG)

        endpoints_rps_resource = sdk2.parameters.Resource('Endpoints RPS resource',
                                                          resource_type=CarterLoadEndpointsRps)
        custom_endpoints_rps = sdk2.parameters.String('Custom endpoints RPS', multiline=True)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.init()
        self.generate_ammo()
        self.generate_config()

    def init(self):
        self.fee_shows = resource_loader.load_fee_shows()
        self.prepare_data()
        self.supported_methods = {
            ('GET', 'carter_yandexuid_light-list'): lambda rgb: (
                self.get_cart_list_light_url(rgb, 'YANDEXUID', self.random_yandexuid_with_basket()),
                None
            ),
            ('GET', 'carter_uuid_light-list'): lambda rgb: (
                self.get_cart_list_light_url(rgb, 'UUID', self.random_uuid_with_basket()),
                None
            ),
            ('GET', 'carter_uid_light-list'): lambda rgb: (
                self.get_cart_list_light_url(rgb, 'UID', self.random_uid_with_basket()),
                None
            ),
            ('GET', 'carter_yandexuid_list'): lambda rgb: (
                self.get_cart_list_url(rgb, 'YANDEXUID', self.random_yandexuid(), False),
                None
            ),
            ('GET', 'carter_yandexuid_list_existing_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'YANDEXUID', self.random_yandexuid_with_basket(), False),
                None
            ),
            ('GET', 'carter_yandexuid_list_nonexisting_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'YANDEXUID', self.random_yandexuid_without_basket(), False),
                None
            ),
            ('GET', 'carter_uuid_list'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UUID', self.generate_random_uuid(), False),
                None
            ),
            ('GET', 'carter_uuid_list_existing_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UUID', self.random_uuid_with_basket(), False),
                None
            ),
            ('GET', 'carter_uuid_list_nonexisting_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UUID', self.random_uuid_without_basket(), False),
                None
            ),
            ('GET', 'carter_uid_list'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UID', self.random_uid(), False),
                None
            ),
            ('GET', 'carter_uid_list_existing_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UID', self.random_uid_with_basket(), False),
                None
            ),
            ('GET', 'carter_uid_list_nonexisting_only'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UID', self.random_uid_without_basket(), False),
                None
            ),
            ('GET', 'carter_yandexuid_list_consolidate'): lambda rgb: (
                self.get_cart_list_url(rgb, 'YANDEXUID', self.random_yandexuid(), True),
                None
            ),
            ('GET', 'carter_uuid_list_consolidate'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UUID', self.generate_random_uuid(), True),
                None
            ),
            ('GET', 'carter_uid_list_consolidate'): lambda rgb: (
                self.get_cart_list_url(rgb, 'UID', self.random_uid(), True),
                None
            ),
            ('GET', 'carter_yandexuid_updated_existing_only'): lambda rgb: (
                self.get_cart_updated_url('YANDEXUID', self.random_yandexuid_with_basket()),
                None
            ),
            ('GET', 'carter_uid_updated_existing_only'): lambda rgb: (
                self.get_cart_updated_url('UID', self.random_uid_with_basket()),
                None
            ),
            ('GET', 'carter_uuid_updated_existing_only'): lambda rgb: (
                self.get_cart_updated_url('UUID', self.random_uuid_with_basket()),
                None
            ),
            ('GET', 'carter_yandexuid_updated_nonexisting_only'): lambda rgb: (
                self.get_cart_updated_url('YANDEXUID', self.random_yandexuid_without_basket()),
                None
            ),
            ('GET', 'carter_uid_updated_nonexisting_only'): lambda rgb: (
                self.get_cart_updated_url('UID', self.random_uid_without_basket()),
                None
            ),
            ('GET', 'carter_uuid_updated_nonexisting_only'): lambda rgb: (
                self.get_cart_updated_url('UUID', self.random_uuid_without_basket()),
                None
            ),

            ('POST', 'carter_yandexuid_list_item'): lambda rgb: (
                self.add_cart_item_url(rgb, 'YANDEXUID', self.random_yandexuid()),
                self.add_cart_item_body()
            ),
            ('POST', 'carter_uuid_list_item'): lambda rgb: (
                self.add_cart_item_url(rgb, 'UUID', self.generate_random_uuid()),
                self.add_cart_item_body()
            ),
            ('POST', 'carter_uid_list_item'): lambda rgb: (
                self.add_cart_item_url(rgb, 'UID', self.random_uid()),
                self.add_cart_item_body()
            ),

            ('POST', 'carter_yandexuid_list_items'): lambda rgb: (
                self.add_cart_items_url(rgb, 'YANDEXUID', self.random_yandexuid()),
                self.add_cart_items_body()
            ),
            ('POST', 'carter_uuid_list_items'): lambda rgb: (
                self.add_cart_items_url(rgb, 'UUID', self.generate_random_uuid()),
                self.add_cart_items_body()
            ),
            ('POST', 'carter_uid_list_items'): lambda rgb: (
                self.add_cart_items_url(rgb, 'UID', self.random_uid()),
                self.add_cart_items_body()
            ),

            ('PATCH', 'carter_yandexuid_list'): lambda rgb: (
                self.merge_cart_url(rgb, 'YANDEXUID', self.random_yandexuid_mutable(),
                                    'YANDEXUID', self.random_yandexuid_mutable()),
                None
            ),
            ('PATCH', 'carter_uid_list'): lambda rgb: (
                self.merge_cart_url(rgb, 'UID', self.random_uid_mutable(),
                                    'YANDEXUID', self.random_yandexuid_mutable()),
                None
            ),

            ('DELETE', 'carter_uid_list_item_by_id'): lambda rgb: (
                self.delete_item_url(rgb, 'UID'),
                None
            ),
            ('DELETE', 'carter_yandexuid_list_item_by_id'): lambda rgb: (
                self.delete_item_url(rgb, 'YANDEXUID'),
                None
            ),

            ('DELETE', 'carter_yandexuid_list_item'): lambda rgb:
            self.delete_items_request(rgb, 'YANDEXUID'),
            ('DELETE', 'carter_uid_list_item'): lambda rgb:
            self.delete_items_request(rgb, 'UID'),

            ('PATCH', 'carter_yandexuid_list_items'): lambda rgb: self.patch_cart_complex(rgb, 'YANDEXUID'),
            ('PATCH', 'carter_uid_list_items'): lambda rgb: self.patch_cart_complex(rgb, 'UID'),

            ('PUT', 'carter_yandexuid_list'): lambda rgb: (
                self.replace_cart_url(rgb, 'YANDEXUID', self.random_yandexuid_mutable()),
                self.add_cart_items_body()
            ),
            ('PUT', 'carter_uid_list'): lambda rgb: (
                self.replace_cart_url(rgb, 'UID', self.random_uid_mutable()),
                self.add_cart_items_body()
            ),

            ('PUT', 'carter_yandexuid_list_item_by_id'): lambda rgb: self.update_item_count_complex(rgb, 'YANDEXUID'),
            ('PUT', 'carter_uuid_list_item_by_id'): lambda rgb: self.update_item_count_complex(rgb, 'UUID'),
            ('PUT', 'carter_uid_list_item_by_id'): lambda rgb: self.update_item_count_complex(rgb, 'UID')
        }

    def prepare_data(self):
        yql_token_vault = sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name)
        yql_token = str(yql_token_vault)
        db = DataBase(yql_token)
        self.uids = resource_loader.load_uids()
        self.yandexuids = resource_loader.load_yandexuids()

        self.yandexuids_with_basket = db.fetch_yandexuids_with_basket()
        self.uids_with_basket = db.fetch_uids_with_basket()
        self.uuids_with_basket = db.fetch_uuids_with_basket()

        self.yandexuids_without_basket = self.generate_yandexuids_without_basket()
        self.uids_without_basket = self.generate_uids_without_basket()
        self.uuids_without_basket = self.generate_uuids_without_basket()

        yandexuid_to_item_id = db.fetch_yandexuid_items()
        self.yandexuid_to_item_id_mutable = yandexuid_to_item_id[:len(yandexuid_to_item_id) / 2]
        self.yandexuid_to_item_id_immutable = yandexuid_to_item_id[len(yandexuid_to_item_id) / 2:]

        uid_to_item_id = db.fetch_uid_items()
        self.uid_to_item_id_mutable = uid_to_item_id[:len(uid_to_item_id) / 2]
        self.uid_to_item_id_immutable = uid_to_item_id[len(uid_to_item_id) / 2:]

        self.create_items_to_delete('UID', db)
        self.uid_to_item_id_to_delete = db.fetch_uid_items_to_delete()
        log.debug('fetched {} uid items to delete'.format(len(self.uid_to_item_id_to_delete)))

    def generate_yandexuids_without_basket(self):
        log.debug('generating YANDEXUIDs without basket')
        yandexuids_with_basket = set(map(lambda e: int(e), self.yandexuids_with_basket))
        return list(set(range(MAX_AMMO_COUNT + len(yandexuids_with_basket))) - yandexuids_with_basket)[:MAX_AMMO_COUNT]

    def generate_uids_without_basket(self):
        log.debug('generating UIDs without basket')
        uids_with_basket = set(map(lambda e: int(e), self.uids_with_basket))
        return list(set(range(MAX_AMMO_COUNT + len(uids_with_basket))) - uids_with_basket)[:MAX_AMMO_COUNT]

    def generate_uuids_without_basket(self):
        log.debug('generating UUIDs without basket')
        uuids_without_basket = []
        for _ in range(MAX_AMMO_COUNT):
            uuids_without_basket.append(self.generate_random_uuid())
        return uuids_without_basket

    def create_items_to_delete(self, user_type, db):
        to_delete_count = len(db.fetch_uid_items_to_delete())
        if to_delete_count >= constants.MAX_ITEMS_TO_DELETE_COUNT:
            return
        else:
            create_max = constants.MAX_ITEMS_TO_DELETE_COUNT - to_delete_count

        log.info('creating items to delete')
        if user_type == 'UID':
            user_free_slots = db.fetch_uids_and_free_item_slots_count(constants.MAX_ITEMS_PER_CART - 30)
        else:
            raise Exception('Unsupported user type: ' + str(user_type))
        created_count = 0
        import carter_client as client
        for row in user_free_slots:
            user_id = row[0]
            free_slots_count = row[1]
            items_to_delete = []
            if free_slots_count > 0:
                for i in range(0, free_slots_count):
                    item = self.generate_item()
                    item['name'] = constants.ITEM_TO_DELETE_PREFIX + item['name']
                    self.append = items_to_delete.append(item)

                client.add_items(user_type, user_id, items_to_delete)
                created_count += free_slots_count
                log.info('created {} items to delete'.format(created_count))
                if created_count >= create_max:
                    break

    def random_yandexuid(self):
        return random.choice(self.yandexuids)

    def random_yandexuid_with_basket(self):
        return random.choice(self.yandexuids_with_basket)

    def random_yandexuid_without_basket(self):
        return random.choice(self.yandexuids_without_basket)

    def random_uid(self):
        return random.choice(self.uids)

    def random_uid_with_basket(self):
        return random.choice(self.uids_with_basket)

    def random_uid_without_basket(self):
        return random.choice(self.uids_without_basket)

    def generate_random_uuid(self):
        return str(uuidm.uuid4())

    def random_uuid_with_basket(self):
        return random.choice(self.uuids_with_basket)

    def random_uuid_without_basket(self):
        return random.choice(self.uuids_without_basket)

    def random_yandexuid_mutable(self):
        return random.choice(self.yandexuid_to_item_id_mutable)[0]

    def random_uid_mutable(self):
        return random.choice(self.uid_to_item_id_mutable)[0]

    def random_yandexuid_item(self):
        entry = random.choice(self.yandexuid_to_item_id_immutable)
        return {'yandexuid': entry[0], 'item_id': entry[1]}

    def random_uid_item(self):
        entry = random.choice(self.uid_to_item_id_immutable)
        return {'uid': entry[0], 'item_id': entry[1]}

    def random_yandexuid_item_mutable(self):
        entry = random.choice(self.yandexuid_to_item_id_mutable)
        return {'yandexuid': entry[0], 'item_id': entry[1]}

    def random_uid_item_mutable(self):
        entry = random.choice(self.uid_to_item_id_mutable)
        return {'uid': entry[0], 'item_id': entry[1]}

    def random_uid_item_to_delete(self):
        entry = random.choice(self.uid_to_item_id_to_delete)
        if len(self.uid_to_item_id_to_delete) > 1:
            self.uid_to_item_id_to_delete.remove(entry)
        else:
            log.warn('no more items to delete!')
        return {'uid': entry[0], 'item_id': entry[1]}

    def get_endpoints_rps_lines(self):
        is_custom_shooting = bool(self.Parameters.custom_endpoints_rps)
        if is_custom_shooting:
            return is_custom_shooting, self.Parameters.custom_endpoints_rps.splitlines()
        else:
            rps_resource_path = str(sdk2.ResourceData(self.Parameters.endpoints_rps_resource).path)
            log.debug('opening endpoints_rps_resource: ' + rps_resource_path)
            with open(rps_resource_path) as rps_file:
                return is_custom_shooting, rps_file.readlines()

    def generate_meta(self):
        is_custom_shooting, rps_lines = self.get_endpoints_rps_lines()
        api_methods = resource_loader.load_api_methods(rps_lines)
        meta = []

        for api_method in api_methods:
            http_method = api_method["http_method"]
            method_name = api_method["method_name"]

            if (http_method, method_name) not in self.supported_methods:
                log.warn('Attention! Method not supported: %s %s' % (http_method, method_name))
                continue

            rps_multiplier = float(api_method["rps_multiplier"])
            rgb = api_method["rgb"]

            meta.append({
                'http_method': http_method,
                'method': method_name,
                'rps_multiplier': rps_multiplier,
                'rgb': rgb,
                'generator': self.supported_methods[(http_method, method_name)]
            })

        return meta

    def generate_ammo(self):
        meta = self.generate_meta()
        log.info('generating ammo')
        max_rounds = MAX_AMMO_COUNT
        if self.Parameters.imbalance_shooting:
            total_rounds = min(max_rounds, self.calculate_total_rounds_for_imbalance_shooting())
        else:
            total_rounds = min(max_rounds, self.Parameters.total_rps * self.Parameters.test_duration_minutes * 60)
        log.info('total rounds: ' + str(total_rounds))
        ammos = []
        total_count = 0
        for meta_line in meta:
            http_method = meta_line["http_method"]
            method = meta_line["method"]
            rgb = meta_line["rgb"]
            generator = meta_line["generator"]
            rps_multiplier = float(meta_line["rps_multiplier"])

            count = max(int(rps_multiplier * total_rounds), 1)
            log.info('count for {} {} = {}'.format(http_method, method, count))
            total_count += count

            for i in range(0, count):
                ammo = self.create_ammo(http_method, method, rgb, generator)
                ammos.append(ammo)
        log.info('total request count: ' + str(total_count))

        random.shuffle(ammos)

        data = sdk2.ResourceData(CarterLoadAmmo(self, 'Carter load ammo', 'ammo.txt'))
        with codecs.open('ammo.txt', 'w', encoding='utf-8') as f:
            for single_ammo in ammos:
                f.write(single_ammo)
        data.ready()

    def calculate_total_rounds_for_imbalance_shooting(self):
        total_rounds = 0
        rps = self.Parameters.start_rps
        while rps <= self.Parameters.target_rps:
            total_rounds += rps * self.Parameters.rps_step_duration_seconds
            rps += self.Parameters.rps_increase_step
        return total_rounds

    def create_ammo(self, http_method, method, rgb, generator):
        url, body = generator(rgb)

        return make_ammo(http_method, url, tag='{}_{}_{}'.format(http_method, method, rgb), body=body)

    def get_cart_list_url(self, rgb, user_type, user_id, consolidate):
        return '/cart/{}/{}/list?rgb={}&consolidate={}'.format(user_type, user_id, rgb, str(consolidate).lower())

    def get_cart_list_light_url(self, rgb, user_type, user_id):
        return '/cart/{}/{}/light-list?rgb={}'.format(user_type, user_id, rgb)

    def get_cart_updated_url(self, user_type, user_id):
        return '/cart/{}/{}/updated'.format(user_type, user_id)

    def merge_cart_url(self, rgb, user_type_from, user_from, user_type_to, user_to):
        return '/cart/{}/{}/list?rgb={}&idTo={}&typeTo={}'.format(user_type_from, user_from, rgb, user_to, user_type_to)

    def delete_item_url(self, rgb, user_type):
        item_info = self.get_item_to_delete(user_type)
        return '/cart/{}/{}/list/-1/item/{}?rgb={}'.format(user_type, item_info[user_type.lower()],
                                                           item_info['item_id'], rgb)

    def get_item_to_delete(self, user_type):
        if user_type == 'UID':
            item_info = self.random_uid_item_to_delete()
        elif user_type == 'YANDEXUID':
            item_info = self.random_yandexuid_item_mutable()
        else:
            raise Exception('Unknown user type: ' + str(user_type))
        return item_info

    def delete_items_request(self, rgb, user_type):
        item_info = self.get_item_to_delete(user_type)
        return ('/cart/{}/{}/list/-1/item?rgb={}'.format(user_type, item_info[user_type.lower()], rgb),
                '[{}]'.format(item_info['item_id']))

    def patch_cart_complex(self, rgb, user_type):
        if user_type == 'YANDEXUID':
            item_info = self.random_yandexuid_item()
            url = self.patch_cart_url(rgb, user_type, item_info['yandexuid'])
            body = self.patch_cart_body(item_info['item_id'])
        elif user_type == 'UID':
            item_info = self.random_uid_item()
            url = self.patch_cart_url(rgb, user_type, item_info['uid'])
            body = self.patch_cart_body(item_info['item_id'])
        else:
            raise Exception('Unknown user type: ' + str(user_type))
        return url, body

    def patch_cart_url(self, rgb, user_type, user_id):
        return '/cart/{}/{}/list/-1/items?rgb={}'.format(user_type, user_id, rgb)

    def patch_cart_body(self, item_id):
        return '''{
        "items": [{
        "id": %d,
        "type": "OFFER",
        "count": 1,
        "fieldsToChange": ["COUNT"]
    }]
}
        ''' % (int(item_id))

    def add_cart_item_url(self, rgb, user_type, user_id):
        return '/cart/{}/{}/list/-1/item?rgb={}'.format(user_type, user_id, rgb)

    def add_cart_item_body(self):
        return json.dumps(self.generate_item())

    def add_cart_items_url(self, rgb, user_type, user_id):
        return '/cart/{}/{}/list/-1/items?rgb={}'.format(user_type, user_id, rgb)

    def add_cart_items_body(self):
        item = self.generate_item()

        body = {
            "listType": "BASKET",
            "items": [item]
        }

        return json.dumps(body)

    def replace_cart_url(self, rgb, user_type, user_id):
        return '/cart/{}/{}/list/-1?rgb={}'.format(user_type, user_id, rgb)

    def update_item_count_url(self, rgb, user_type, user_id, item_id):
        return '/cart/{}/{}/list/-1/item/{}?rgb={}&count=1'.format(user_type, user_id, item_id, rgb)

    def update_item_count_complex(self, rgb, user_type):
        if user_type == 'YANDEXUID':
            item_info = self.random_yandexuid_item()
            url = self.update_item_count_url(rgb, user_type, item_info['yandexuid'], item_info['item_id'])
        elif user_type == 'UID':
            item_info = self.random_uid_item()
            url = self.update_item_count_url(rgb, user_type, item_info['uid'], item_info['item_id'])
        else:
            raise Exception('Unknown user type: ' + str(user_type))
        return url, None

    def generate_item(self):
        return {
            "objType": "OFFER",
            "objId": str(random.randint(1, 10000000)),
            "msku": random.randint(1, 10000000),
            "name": ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(20)),
            "shopId": random.randint(1, 100000),
            "count": 1,
            "hid": random.randint(1, 100000),
            "feeShow": random.choice(self.fee_shows),
            "price": random.randint(1, 10000)
        }

    def generate_config(self):
        config_file_name = 'check.yaml'
        config_resource = sdk2.ResourceData(CarterLoadConfig(self, 'Carter load config', config_file_name))

        config = self.read_template()
        if self.Parameters.imbalance_shooting:
            config = self.generate_config_lines_for_imbalance_shooting(config)
        else:
            config = self.generate_config_lines_for_const_shooting(config)

        with codecs.open(config_file_name, 'w', encoding='utf-8') as f:
            f.write(config)
        config_resource.ready()

    def read_template(self):
        from library.python import resource
        return resource.find('sandbox/projects/market/checkout/MarketCarterLoadGenerateAmmo/config/check.yaml.tmpl')

    def generate_config_lines_for_const_shooting(self, config):
        log.info('Adding config lines for CONST shooting')
        rps = int(self.Parameters.total_rps)
        start_rps = max(int(rps * 0.05), 1)
        rps_schedule = 'const({}, 5m) line({},{},2m) const({}, {}m)' \
            .format(start_rps, start_rps, rps, rps, self.Parameters.test_duration_minutes)
        return config.format(schedule=rps_schedule, autostop_enabled='false',
                             autostop_config='')

    def generate_config_lines_for_imbalance_shooting(self, config):
        log.info('Adding config lines for IMBALANCE shooting')
        rps_schedule = 'const(1, 10s) step({}, {}, {}, {}s)'.format(self.Parameters.start_rps,
                                                                    self.Parameters.target_rps,
                                                                    self.Parameters.rps_increase_step,
                                                                    self.Parameters.rps_step_duration_seconds)
        log.info('Using autostop config: ' + str(self.Parameters.autostop_config))
        autostop_config = '- ' + '\n  - '.join(self.Parameters.autostop_config.split(' '))
        return config.format(schedule=rps_schedule, autostop_enabled='true',
                             autostop_config=autostop_config)
