from collections import defaultdict

import logging
import requests
from sandbox import sdk2


def auth_headers(auth_token):
    return {"Authorization": "OAuth " + auth_token, "Content-Type": "application/json; charset=utf-8"}


class SolomonPusher(object):

    def __init__(self, solomon_url, auth_token, project_id, service_id):
        self.solomon_url = solomon_url
        self.auth_token = auth_token
        self.project_id = project_id
        self.service_id = service_id

    def post(self, data):
        url = '%s/api/v2/push?project=%s&cluster=testing&service=%s' % (
            self.solomon_url, self.project_id, self.service_id)
        return requests.post(url, headers=auth_headers(self.auth_token), json=data)


class TestResult(object):

    def __init__(self, sandbox_time, time_by_chunk, count_by_chunk, time_by_test, count_by_test, test_statuses,
                 test_files_count):
        """
        Parameters
        ----------
        sandbox_time: float
            Sandbox task execution time
        time_by_chunk: dict[str, float]
            Test execution time by chunk
            Essentially shows the run time of an every chunk (chunk initialization + spring context load + test time)
        count_by_chunk: dict[str, int]
            Amount of tests in every chunk
        time_by_test: dict[str, float]
            Test execution time by test
            Essentially shows the run time of every test (without spring context load - only pure test time)
        count_by_test: dict[str, int]
            Amount of tests in every chunk (with expanding parametrized)
        test_statuses: dict[str, int]
            Amount of tests by statuses
        test_files_count: int
            Amount of test files
        """
        self.sandbox_time = sandbox_time
        self.time_by_chunk = time_by_chunk
        self.count_by_chunk = count_by_chunk
        self.time_by_test = time_by_test
        self.count_by_test = count_by_test
        self.test_statuses = test_statuses
        self.test_files_count = test_files_count


class ResultsParser(object):

    def parse(self, content):
        time_by_chunk = defaultdict(float)
        count_by_chunk = defaultdict(int)
        time_by_test = defaultdict(float)
        count_by_test = defaultdict(int)
        test_file_count = 0

        test_statuses = defaultdict(int)
        results = content['results']
        logging.info('results.keys count: %d' % len(results.keys()))

        chunk_by_id = dict()
        for result in results:
            if result.get('type') == 'test':
                if result.get('chunk'):
                    if result['subtest_name'] != 'sole chunk':
                        chunk = int(result['subtest_name'][1:].split('/')[0])
                        chunk_duration = result['duration']
                        time_by_chunk[chunk] += chunk_duration
                        chunk_by_id[result['id']] = chunk
                else:
                    if 'chunk_id' in result:
                        name = result['name']
                        test_duration = result['duration']
                        time_by_test[name] += test_duration

                if result.get('suite'):
                    test_file_count += 1

                test_statuses[result['status']] += 1

        progress = content['progress']
        logging.info('progress.keys count: %d' % len(progress.keys()))
        sandbox_time = 0.0
        for k, v in progress.items():
            for e in v:
                if 'total' in e:
                    sandbox_time += e['total']

        return TestResult(sandbox_time, time_by_chunk, count_by_chunk, time_by_test, count_by_test, test_statuses,
                          test_file_count)


class MarketCheckouterAnalyzeTests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        test_resource = sdk2.parameters.String('Test resource')
        solomon_url = sdk2.parameters.String('Solomon api url', default='http://solomon.yandex.net')
        project_id = sdk2.parameters.String('Project id', default='market-checkout')
        service_id = sdk2.parameters.String('Service id', default='market-checkouter')
        auth_token = sdk2.parameters.String('Oauth token vault key', default='market_checkouter_solomon_oauth_token')

    def on_execute(self):
        params = self.Parameters
        auth_token = sdk2.Vault.data('MARKET', params.auth_token)

        pusher = SolomonPusher(params.solomon_url, auth_token, params.project_id, params.service_id)

        log_data = self.download_log(auth_token)

        times = ResultsParser().parse(log_data)
        data = self.build_metrics(times)
        logging.info("Data for push to solomon is: %s", data)

        result = pusher.post(data)
        logging.info("Push to solomon result: %s", result)

    def build_metrics(self, times):
        pure_test_metrics = [
            {
                "labels": {
                    "sensor": "timings",
                    "timings": "pure",
                    "name": test_name
                },
                "value": times.time_by_test[test_name]
            } for test_name in times.time_by_test]

        chunk_test_metrics = [
            {
                "labels": {
                    "sensor": "timings",
                    "timings": "chunk",
                    "name": str(chunk_name)
                },
                "value": times.time_by_chunk[chunk_name]
            } for chunk_name in times.time_by_chunk]

        metrics = [
            {
                "labels": {
                    "sensor": "timings",
                    "timings": "sandbox"
                },
                "value": times.sandbox_time
            }

        ]
        metrics.extend(pure_test_metrics)
        metrics.extend(chunk_test_metrics)

        return {
            "metrics": metrics
        }

    def download_log(self, auth_token):
        url = 'https://proxy.sandbox.yandex-team.ru/task/%s/results.json' % self.Parameters.test_resource
        logging.info('Downloading log file...: %s' % url)
        log_file = requests.get(url, headers=auth_headers(auth_token))
        logging.debug(log_file.content)
        logging.debug('Encoding is: %s' % log_file.encoding)
        result = log_file.json()
        logging.info('results.json, keys count is: %d' % len(result.keys()))
        return result
