# coding=UTF8
import requests
import json
from datetime import timedelta
import logging


class StartrekProvider():
  def __init__(self,
               startrek_oauth_token,
               startrek_url='https://st-api.yandex-team.ru/v2'):
    self.startrek_url = startrek_url
    self.startrek_oauth_token = startrek_oauth_token
    self.logger = logging.getLogger(__name__)

  def auth_header(self):
    return {'Authorization': 'Bearer {}'.format(self.startrek_oauth_token), 'Content-Type': 'application/json'}

  def search_request(self, from_date, to_date):
    return {
      "filter": {
        "queue": "MARKETCHECKOUT",
        "summary": "Факап дежурство",
        "tags": ["checkouter_fm_duty"],
        "start": from_date.strftime('%Y-%m-%d'),
        "end": to_date.strftime('%Y-%m-%d')
      }
    }

  def actualize_issues_status(self):
    search_request = "Queue%3A+MARKETCHECKOUT" \
                     "+Summary%3A+%22Факап+дежурство%22" \
                     "+Resolution%3A+empty%28%29" \
                     "+Tags%3A+checkouter_fm_duty" \
                     "+%22End+Date%22%3A+<=+today%28%29"
    issues = requests.get(
      '{}/issues?query={}'.format(self.startrek_url, search_request),
      headers=self.auth_header()
    ).json()
    self.logger.info("Found issues to actualize {}".format(issues))

    transition_request = {
      "resolution": "fixed"
    }
    for issue in issues:
      response = requests.post(
        '{}/issues/{}/transitions/close/_execute'.format(self.startrek_url, issue["key"]),
        headers=self.auth_header(),
        data=json.dumps(transition_request)
      ).json()
      self.logger.info("Closing issue {} result {}".format(issue, response))

  def create_divided_duty_issues_from_duty_row(self, row):
    issue = self.create_issue(row['from_date'], row['from_date'] + timedelta(days=4), row['duty'][0])
    self.create_issue(
      row['from_date'] + timedelta(days=4),
      row['to_date'],
      row['duty'][2],
      issue=issue
    )

  def update_issue(self, duty, issue_key):
    response = requests.patch(
      '{}/issues/{}'.format(self.startrek_url, issue_key),
      headers=self.auth_header(),
      data=json.dumps({
        "assignee": duty,
        "summary": "Факап дежурство {}".format(duty)
      })
    )
    logging.debug("Received update issue response {}".format(str(response)))

    return response.json()

  def create_issue(self, from_date, to_date, duty, issue=None):
    found_issues = self.post_startrek_search_request(self.search_request(from_date, to_date))
    self.logger.debug("Found duty issues {}".format(str(found_issues)))

    if not len(found_issues):
      create_issue_request = self.create_issue_request(from_date, to_date, duty, issue)
      self.logger.debug("Sending create issue request {}".format(str(create_issue_request)))

      response = requests.post(
        '{}/issues'.format(self.startrek_url),
        headers=self.auth_header(),
        data=json.dumps(create_issue_request)
      )
      self.logger.debug("Received create issue response {}".format(str(response)))

      return response.json()
    else:
      for issue in found_issues:
        if "assignee" not in issue or issue["assignee"] != duty:
          return self.update_issue(duty, issue["key"])
        else:
          return issue

  def post_startrek_search_request(self, request):
    return requests.post(
      '{}/issues/_search'.format(self.startrek_url),
      headers=self.auth_header(),
      data=json.dumps(request)
    ).json()

  def get_startrek_issues(self, request):
    return requests.get(
      '{}/issues?query={}'.format(self.startrek_url, request),
      headers=self.auth_header()
    ).json()

  def create_issue_request(self, from_date, to_date, duty, issue=None):
    summary = "Факап дежурство {}".format(duty)
    create_issue_request = {
      "queue": "MARKETCHECKOUT",
      "summary": summary,
      "tags": ["checkouter_fm_duty"],
      "start": from_date.strftime('%Y-%m-%d'),
      "end": to_date.strftime('%Y-%m-%d'),
      "assignee": duty,
      "links": []
    }
    if issue:
      create_issue_request["links"].append({
        "issue": issue["key"],
        "relationship": "RELATES"
      })

    return create_issue_request
