import logging
import os
import requests

class DashboardGenerator(object):
    def __init__(self,
                    checkouter_url,
                    application_path,
                    project_id,
                    annotation_keys,
                    solomon,
                    dashboard_id,
                    solomon_url = 'https://solomon.yandex-team.ru',
                    row_size = 5,
                    annotation_key_filter = {},
                    height_multiplier = 1):
        self.checkouter_url = checkouter_url
        self.row_size  = row_size
        self.application_path = application_path
        self.annotation_keys = annotation_keys
        self.solomon = solomon
        self.dashboard_id = dashboard_id
        self.solomon_url = solomon_url
        self.project_id = project_id
        self.annotation_key_filter = annotation_key_filter
        self.height_multiplier = height_multiplier

    def generate(self):
        elements = self.__generate_elements()

        row_size = 0
        file_name = self.application_path[1:].replace('/', '_')

        rows = []
        row = {
            "panels": []
        }
        for element in elements:
            row['panels'].append(
                {
                    "url": element['url'],
                    "type": "IFRAME",
                    "title": ""
                }
            )

            if len(row['panels']) == self.row_size:
                rows.append(row)
                row = {
                    "panels": []
                }

        if len(row['panels']) > 0:
            rows.append(row)

        return rows

    def __generate_elements(self):
        alerts = requests.get(self.checkouter_url + self.application_path, headers={"IgnoreTvmCheck": "true"}).json()

        elements = []
        alerts.sort(key=lambda  a: a['id'])
        for alert in alerts:
            url = "{solomon_url}/admin/projects/{project_id}/alerts/{alert_id}/singlestat/fullscreen?annotationKeys={annotation_keys}".format(
                solomon_url = self.solomon_url,
                alert_id = alert['id'],
                project_id = self.project_id,
                annotation_keys = self.annotation_keys
            )

            if len(self.annotation_key_filter) > 0:
                skip = False
                for k, v in self.annotation_key_filter.iteritems():
                    if k not in alert['annotations'] or alert['annotations'][k] != v:
                        skip = True
                if skip:
                    continue

            elements.append({
                "url": url,
                "title": ""
            })

        return elements

    def update_dashboard(self, elements):
        logging.info('project_id: %s', self.project_id)
        dashboard = self.solomon.get_dashboard(self.project_id, self.dashboard_id)

        body = {
            "name": self.dashboard_id,
            "rows": elements,
            "heightMultiplier": self.height_multiplier
        }

        if dashboard is None:
            body['id'] = self.dashboard_id
            self.solomon.create_dashboard(self.project_id, self.dashboard_id, body)
        else:
            body['version'] = dashboard['version']
            self.solomon.update_dashboard(self.project_id, self.dashboard_id, body)


def main():
    import os,sys
    currentdir = os.path.dirname(os.path.abspath(__file__))
    parentdir = os.path.dirname(currentdir)
    sys.path.insert(0,parentdir)

    from helpers.solomon import Solomon

    logging.basicConfig(level=logging.DEBUG)

    oauthToken = os.environ['SOLOMON_OAUTH']

    generator = DashboardGenerator(
        'http://checkouter.tst.vs.market.yandex.net:39001',
        '/tasks/health/solomon',
        'market-checkouter-devel',
        'taskName',
        Solomon(oauthToken),
        'market-checkouter-tasks-alerts',
        row_size=4
    )
    result = generator.generate()
    generator.update_dashboard(result)

if __name__ == '__main__':
    main()
